#include <string>
#include <sstream>
#include <iostream>
#include <gtkmm/arrow.h>
#include <gtkmm/stock.h>
#include <gtkmm/image.h>

using namespace std;
#include "gtkplaylistnotebook.h"
#include "facontext.h"
#include "playlist.h"
#include "gtkmusicbrowser.h"
#include "gtkplaylist.h"
#include "musiclayout.h"

GTKPlaylistNotebook::GTKPlaylistNotebook(FAContext *context, PlaylistManager *CurrentPlaylist, GTKBrowserMusic *musicbrowser)
	: Notebook()
{
	//Set Style
	set_show_border(true);
	set_border_width(6);
	set_scrollable(true);
	popup_enable();
	
	//Init Variables
	m_context = context;
	m_musicbrowser = musicbrowser;
	new_playlist_count = 0;
	
	//Create Main Tab
	AddPlaylist(CreatePlaylist(CurrentPlaylist), "Currently Listening To");
}

GTKPlaylistNotebook::~GTKPlaylistNotebook()
{
	std::vector<GTKPlaylist *>::iterator Playlist;
	std::vector<GTKTab *>::iterator Tab;
	
	for(Playlist = m_playlists.begin(); Playlist != m_playlists.end(); Playlist++) {
		PlaylistManager *OldPLM;
		OldPLM = (*Playlist)->GetPLM();
		delete (*Playlist);
		if(OldPLM->GetActivePlaylist() != kPlaylistKey_MasterPlaylist)
			delete OldPLM;
	}
	
	for(Tab = m_tabs.begin(); Tab != m_tabs.end(); Tab++) {
		delete *Tab;
	}
}

void GTKPlaylistNotebook::set_layout(MusicLayout *layout)
{
	if(layout->get_tab_position() != -1) {
		set_tab_pos((Gtk::PositionType)layout->get_tab_position());
		set_show_tabs(true);
	} else
		set_show_tabs(false);
}

Error GTKPlaylistNotebook::AcceptEvent(Event *event)
{
	std::vector<GTKPlaylist *>::iterator Playlist;
	
	//Send Message To All Playlists
	for(Playlist = m_playlists.begin(); Playlist != m_playlists.end(); Playlist++)
		(*Playlist)->AcceptEvent(event);
	
	return kError_NoErr;
}

void GTKPlaylistNotebook::on_switch_page(GtkNotebookPage* page, guint page_num)
{
	Gtk::Notebook::on_switch_page(page, page_num);
	
	//Tell GTKBrowserMusic Current PLM
	m_musicbrowser->SetCurrentPLM( ((GTKPlaylist *)get_nth_page(page_num))->GetPLM() );
	m_musicbrowser->UpdateListTime();
}

GTKPlaylist* GTKPlaylistNotebook::CreatePlaylist(PlaylistManager *NewPLM)
{
	PlaylistManager *new_plm;
	GTKPlaylist *new_list;
	
	//Create New Playlist Manager
	if(!NewPLM) {
		new_plm = new PlaylistManager(m_context);
		new_plm->SetActivePlaylist(kPlaylistKey_ExternalPlaylist);
	}
	else {
		new_plm = NewPLM;
	}
	
	//Create New GTKPlaylist
	new_list = new GTKPlaylist(m_context, new_plm, m_musicbrowser);
	m_playlists.push_back(new_list);

	new_list->show_all();
	
	return new_list;
}

int GTKPlaylistNotebook::AddPlaylist(GTKPlaylist *playlist, Glib::ustring Caption)
{
	GTKTab *new_tab;
	
	//Create New Tab
	new_tab = new GTKTab(this, Caption, playlist);
	m_tabs.push_back(new_tab);
	
	//Add New Page To Notebook
	append_page(*playlist, *new_tab);
	set_menu_label_text(*playlist, Caption);
	
	return get_n_pages() - 1;
}
	

void GTKPlaylistNotebook::NewPlaylist()
{
	//Add To Notebook
	ostringstream NewTitle;
	NewTitle << "New Playlist #" << ++new_playlist_count;
	set_current_page(AddPlaylist(CreatePlaylist(), NewTitle.str()));
}

void GTKPlaylistNotebook::OpenPlaylist(std::string Filename)
{
	GTKPlaylist *new_list;
	PlaylistManager *new_plm;
	
	//Load Playlist
	new_list = CreatePlaylist();
	new_list->SetLoadingPlaylist();
	new_plm = new_list->GetPLM();
	new_plm->ReadPlaylist(Filename.c_str());
	
	//Add To Notebook
	int new_page;
	new_page = AddPlaylist(new_list, Filename);
	((GTKTab *)get_tab_label(*new_list))->SetFilename(Filename);
	
	//Switch To New Tab
	set_current_page(new_page);
}

void GTKPlaylistNotebook::SavePlaylist(int playlist_num, bool in_main)
{
	//Get Playlist Number
	if(playlist_num == -1) //None Specified, So Use Current
		playlist_num = get_current_page();
	
	//Get Playlist
	GTKPlaylist *CurrentList;
	PlaylistManager *CurrentPLM;
	GTKTab *CurrentTab;
	std::string Filename;
	CurrentList = (GTKPlaylist *)get_nth_page(playlist_num);
	CurrentTab = (GTKTab *)get_tab_label(*CurrentList);
	
	//Save Playlist
	CurrentPLM = CurrentList->GetPLM();
	if(CurrentPLM->GetActivePlaylist() == kPlaylistKey_MasterPlaylist) {
		//Always Call Save As For Master Playlist
		m_musicbrowser->SavePlaylistAs(playlist_num, in_main);
		return;
	}
	
	Filename = CurrentTab->GetFilename();
	if(Filename.length() > 0) {
		CurrentPLM->WritePlaylist(Filename.c_str());
		CurrentList->SetPlaylistModified(false);
	} else {
		//Call Save Playlist As
		m_musicbrowser->SavePlaylistAs(playlist_num, in_main);
	}
}

void GTKPlaylistNotebook::SavePlaylistAs(std::string Filename, int playlist_num)
{
	//Get Playlist Number
	if(playlist_num == -1) //None Specified, So Use Current
		playlist_num = get_current_page();
	
	//Get Current Playlist
	GTKPlaylist *CurrentList;
	PlaylistManager *CurrentPLM;
	GTKTab *CurrentTab;
	CurrentList = (GTKPlaylist *)get_nth_page(playlist_num);
	CurrentTab = (GTKTab *)get_tab_label(*CurrentList);
	
	//Save Playlist
	CurrentPLM = CurrentList->GetPLM();
	if(CurrentPLM->GetActivePlaylist() != kPlaylistKey_MasterPlaylist) { //Master Playlist Dosn't Have Associated Filename
		CurrentTab->SetFilename(Filename);
	}
	CurrentPLM->WritePlaylist(Filename.c_str());
	CurrentList->SetPlaylistModified(false);
}

void GTKPlaylistNotebook::RevertToSavedPlaylist(int playlist_num)
{
	//Get Playlist Number
	if(playlist_num == -1) //None Specified, So Use Current
		playlist_num = get_current_page();
	
	//Get Playlist
	GTKPlaylist *CurrentList;
	PlaylistManager *CurrentPLM;
	GTKTab *CurrentTab;
	CurrentList = (GTKPlaylist *)get_nth_page(playlist_num);
	CurrentTab = (GTKTab *)get_tab_label(*CurrentList);
	
	//If Filename Associated, Reload It
	if(CurrentTab->GetFilename().length() > 0) {
		CurrentPLM = CurrentList->GetPLM();
		//Make Sure We Think Playlist Hasn't Changed
		CurrentList->SetPlaylistModified(false);
		CurrentList->SetLoadingPlaylist();
		//Actually Load Playlist
		CurrentPLM->RemoveAll();
		CurrentPLM->ReadPlaylist(CurrentTab->GetFilename().c_str());
	}
}

void GTKPlaylistNotebook::ClearPlaylist(int playlist_num)
{
	//Get Playlist Number
	if(playlist_num == -1) //None Specified, So Use Current
		playlist_num = get_current_page();
	
	//Get Current Playlist
	GTKPlaylist *CurrentList;
	PlaylistManager *CurrentPLM;
	CurrentList = (GTKPlaylist *)get_nth_page(playlist_num);
	CurrentPLM = CurrentList->GetPLM();
	
	CurrentPLM->RemoveAll();
}

bool GTKPlaylistNotebook::CheckClosePlaylist(int playlist_num)
{	
	//Get Playlist Number
	if(playlist_num == -1) //None Specified, So Use Current
		playlist_num = get_current_page();

	//Check If Playlist Has Changed, Return False If So
	if(!playlist_num == 0) {
		//Get Playlist
		GTKPlaylist *Playlist;
		
		Playlist = (GTKPlaylist *)get_nth_page(playlist_num);
		return !(Playlist->GetPlaylistModified());
	} else
		return false; //Main Playlist is Never Closed
}
	
void GTKPlaylistNotebook::ClosePlaylist(int playlist_num)
{	
	//Get Playlist Number
	if(playlist_num == -1) //None Specified, So Use Current
		playlist_num = get_current_page();
	
	//Remove Playlist
	if(!playlist_num == 0) { //Don't Delete Main Playlist
		//Get Playlist To Delete
		GTKPlaylist *OldPlaylist;
		PlaylistManager *OldPLM;
		GTKTab *OldTab;
		OldPlaylist = (GTKPlaylist *)get_nth_page(playlist_num);
		OldPLM = OldPlaylist->GetPLM();
		OldTab = (GTKTab *)get_tab_label(*OldPlaylist);
		
		//Remove From Notebook
		remove_page(playlist_num);
		
		//Remove Playlist From List
		std::vector<GTKPlaylist *>::iterator Playlist;
		Playlist = m_playlists.begin();
		while((*Playlist) != OldPlaylist)
			Playlist++;
		m_playlists.erase(Playlist);
		//Remove Tab From List
		std::vector<GTKTab *>::iterator Tab;
		Tab = m_tabs.begin();
		while((*Tab) != OldTab)
			Tab++;
		m_tabs.erase(Tab);
		
		//Delete Objects
		delete OldPlaylist;
		if(OldPLM->GetActivePlaylist() != kPlaylistKey_MasterPlaylist)
			delete OldPLM;
		delete OldTab;
	}
}

void GTKPlaylistNotebook::SetCanClosePlaylist(int playlist_num)
{
	//Get Playlist Number
	if(playlist_num == -1) //None Specified, So Use Current
		playlist_num = get_current_page();
	
	GTKPlaylist *CurrentList;
	CurrentList = (GTKPlaylist *)get_nth_page(playlist_num);
	
	//Set Playlist Hasn't Been Modified So It Can Be Closed
	CurrentList->SetPlaylistModified(false);
}

void GTKPlaylistNotebook::CheckCurrentIndex()
{
	//Get Current Playlist
	GTKPlaylist *CurrentList;
	PlaylistManager *CurrentPLM;
	CurrentList = (GTKPlaylist *)get_nth_page(get_current_page());
	CurrentPLM = CurrentList->GetPLM();

	//Check Current Index If Master Playlist
	if(CurrentPLM->GetActivePlaylist() == kPlaylistKey_MasterPlaylist) { 
		CurrentList->CheckCurrentIndex();
	}
}

Glib::ustring GTKPlaylistNotebook::get_tab_label_text(int page_num)
{
	GTKPlaylist *playlist;
	GTKTab *tab;
	
	if(page_num == -1)
		page_num = get_current_page();
	
	//Get Playlist And Associated Tab
	playlist = (GTKPlaylist *)get_nth_page(page_num);
	tab = (GTKTab *)get_tab_label(*playlist);
	
	return tab->get_text();
}

void GTKPlaylistNotebook::show_page(int page_num)
{
	if(page_num != -1) //-1 Means Current Page
		set_current_page(page_num);
}

bool GTKPlaylistNotebook::has_focus()
{
	//Get Current Playlist
	GTKPlaylist *CurrentList;
	CurrentList = (GTKPlaylist *)get_nth_page(get_current_page());
	
	//Return if current playlist has focus, or notebook
	return (CurrentList->has_focus() || Notebook::has_focus());
}

void GTKPlaylistNotebook::DoEditTrack()
{
	//Get Current Playlist
	GTKPlaylist *CurrentList;
	CurrentList = (GTKPlaylist *)get_nth_page(get_current_page());
	
	//Edit Track
	CurrentList->DoEditTrack();
}

void GTKPlaylistNotebook::DoRemove()
{
	//Get Current Playlist
	GTKPlaylist *CurrentList;
	CurrentList = (GTKPlaylist *)get_nth_page(get_current_page());
	
	//Remove Track(s)
	CurrentList->DoRemove();
}

GTKPlaylistNotebook::GTKTab::GTKTab(GTKPlaylistNotebook *notebook, Glib::ustring Text, GTKPlaylist* child)
{
	//Init Variables
	m_notebook = notebook;
	m_child = child;
	
	//Set Label
	m_TabLabel.set_label(Text);

	//Create Button
	//Get Label Height So Button Isn't Too Big
	GtkRequisition label_size;
	m_TabLabel.size_request(&label_size);
	//Create Arrow Pic
	Gtk::Arrow *arrow = Gtk::manage( new Gtk::Arrow(Gtk::ARROW_RIGHT, Gtk::SHADOW_NONE));
	arrow->set_size_request(-1, label_size.height / 2); //4 * 3);
	m_TabButton.add(*arrow);
	m_TabButton.set_relief(Gtk::RELIEF_NONE);
	//Connect Signal Handler
	m_TabButton.signal_button_release_event().connect( SigC::slot(*this, &GTKTab::on_show_menu), false );
	
	
	//Add Widgets To Tab
	pack_start(m_TabLabel, true, true, 0);
	pack_start(m_TabButton, false, false, 0);
	
	
	//Create Menu
	Gtk::Menu_Helpers::MenuList MenuItems = m_TabMenu.items();
	
	MenuItems.push_back( Gtk::Menu_Helpers::StockMenuElem(Gtk::Stock::CLOSE, SigC::slot(*this, &GTKTab::on_menu_close) ));
	MenuItems.push_back( Gtk::Menu_Helpers::SeparatorElem() );
	Gtk::Image* PlayImage = Gtk::manage( new Gtk::Image(Gtk::Stock::APPLY, Gtk::ICON_SIZE_MENU));
	MenuItems.push_back( Gtk::Menu_Helpers::ImageMenuElem("Play Now" , *PlayImage, SigC::slot(*this, &GTKTab::on_menu_play) ));
	MenuItems.push_back( Gtk::Menu_Helpers::StockMenuElem(Gtk::Stock::CLEAR, SigC::slot(*this, &GTKTab::on_menu_clear) ));
	MenuItems.push_back( Gtk::Menu_Helpers::StockMenuElem(Gtk::Stock::SAVE, SigC::slot(*this, &GTKTab::on_menu_save) ));
	MenuItems.push_back( Gtk::Menu_Helpers::StockMenuElem(Gtk::Stock::SAVE_AS, SigC::slot(*this, &GTKTab::on_menu_save_as) ));
	MenuItems.push_back( Gtk::Menu_Helpers::StockMenuElem(Gtk::Stock::REVERT_TO_SAVED, SigC::slot(*this, &GTKTab::on_menu_revert) ));
	
	//Disable Menu Items Not Valid Without A Filename
	MenuItems[6].set_sensitive(false); //Disable Revert
	
	//Disbale Menu Items Not Valid For Main Playlist
	if(m_child->GetPLM()->GetActivePlaylist() == kPlaylistKey_MasterPlaylist) {
		MenuItems[0].set_sensitive(false); //Disable Close
		MenuItems[2].set_sensitive(false); //Disable Play Now
		MenuItems[6].set_sensitive(false); //Disable Revert
	}
	
	//Show Tab
	show_all();
}
	
void GTKPlaylistNotebook::GTKTab::SetFilename(std::string Filename)
{
	m_filename = Filename;
	
	m_TabLabel.set_label(Glib::path_get_basename(Filename));
	
	//Enable Menu Items Now Valid With Filename
	Gtk::Menu_Helpers::MenuList MenuItems = m_TabMenu.items();
	MenuItems[6].set_sensitive(true); //Enable Revert
	
	//Change Menu Label Name
	m_notebook->set_menu_label_text(*m_child, Filename);
}

bool GTKPlaylistNotebook::GTKTab::on_show_menu(GdkEventButton *event)
{
	m_TabMenu.popup(event->button, event->time);
	return false;
}

void GTKPlaylistNotebook::GTKTab::on_menu_close()
{
	m_notebook->m_musicbrowser->ClosePlaylist(page());
}

void GTKPlaylistNotebook::GTKTab::on_menu_play() 
{
	PlaylistManager *this_playlist;
	PlaylistManager *main_playlist;
	
	//Get Playlists
	this_playlist = m_child->GetPLM();
	main_playlist = m_notebook->m_context->plm;
	
	if(this_playlist == main_playlist) //Check this is not main playlist
		return;
	
	//Clear Main Playlist
	main_playlist->RemoveAll();
	
	//Add Tracks To Playlist
	uint32_t track_index;
	for(track_index = 0; track_index < this_playlist->CountItems(); track_index++)
		main_playlist->AddItem(this_playlist->ItemAt(track_index)->URL());
	
	//Switch To Main Playlist
	m_notebook->set_current_page(0);
	
	//Play New Tracks
	m_notebook->m_musicbrowser->StartPlaying();
}

void GTKPlaylistNotebook::GTKTab::on_menu_clear()
{
	//Switch To This Tab
	m_notebook->set_current_page(page());
	
	m_notebook->m_musicbrowser->ClearPlaylist(page());
}

void GTKPlaylistNotebook::GTKTab::on_menu_save()
{
	//Switch To This Tab
	m_notebook->set_current_page(page());
	
	m_notebook->m_musicbrowser->SavePlaylist(page());
}

void GTKPlaylistNotebook::GTKTab::on_menu_save_as()
{
	//Switch To This Tab
	m_notebook->set_current_page(page());

	m_notebook->m_musicbrowser->SavePlaylistAs(page());
}

void GTKPlaylistNotebook::GTKTab::on_menu_revert()
{
	//Switch To This Tab
	m_notebook->set_current_page(page());

	m_notebook->m_musicbrowser->RevertToSavedPlaylist(page());
}
/* arch-tag: fc1b9c59-9276-48ef-905e-dbbd7158654c */
