
/**********************************************************************
zyGrib: meteorological GRIB file viewer
Copyright (C) 2008-2010 - Jacques Zaninetti - http://www.zygrib.org

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
***********************************************************************/

#ifndef GRIDDEDPLOTTER_H
#define GRIDDEDPLOTTER_H

#include <iostream>
#include <cassert>
#include <cmath>
#include <vector>
#include <set>
#include <map>

#include <QApplication>
#include <QPainter>

#include "DataMeteoAbstract.h"
#include "GriddedReader.h"
#include "Projection.h"
#include "IsoLine.h"
#include "Util.h"

//===============================================================
class GriddedPlotter : public DataPlotterAbstract 
{
    public:
		GriddedPlotter ();
		virtual ~GriddedPlotter () {};
		
		virtual bool  isReaderOk () const = 0;
		virtual GriddedReader *getReader () const = 0;
		virtual void  loadFile (QString fileName) = 0;

		//----------------------------------------------------------------
		// Data manipulation
		//----------------------------------------------------------------
        virtual void     setCurrentDate (time_t t) = 0;
        virtual void     setCurrentDateClosestFromNow ();
        virtual time_t   getCurrentDate () const    {return currentDate;}
        virtual time_t   getClosestDateFromNow  () const
							{ return isReaderOk() ? 
								getReader()->getClosestDateFromNow():0;}
		
        virtual std::set<time_t> * getListDates ()  {return &listDates;}
        virtual int getNumberOfDates ()  {return listDates.size();}
		
		virtual bool hasData (const DataCode &dtc) const
						{ return isReaderOk() && getReader()->hasData(dtc); }
		virtual bool hasData (int dataType, int levelType, int levelValue)  const
						{ return hasData (DataCode(dataType,levelType,levelValue)); }
		virtual bool hasDataType (int dataType)  const
						{ return isReaderOk() && getReader()->hasDataType(dataType); }
		virtual std::set<Altitude> getAllAltitudes (int dataType) const; 
		virtual std::set<DataCode> getAllDataCode ()  const;
		
		//----------------------------------------------------------------
		// Drawing parameters
		//----------------------------------------------------------------
		virtual void setInterpolateValues (bool b)
							{mustInterpolateValues = b;}
		virtual void setWindArrowsOnGrid  (bool b)
							{drawWindArrowsOnGrid = b;}
		virtual void duplicateFirstCumulativeRecord (bool b)
							{mustDuplicateFirstCumulativeRecord = b;}

		virtual void    setIsobarsStep (double step)
							{isobarsStep = step;}
        virtual double  getIsobarsStep () 
							{return isobarsStep;}
							
        virtual void    setIsotherms0Step (double step)
							{isotherms0Step = step;}
        virtual double  getIsotherms0Step () 
							{return isotherms0Step;}
							
        virtual Altitude getWindAltitude () 
							{return windAltitude;}
		
		//----------------------------------------------------------------
		// Drawing functions (virtual not pure)
		//----------------------------------------------------------------
		/** Data: write numerical values on the map (temperature).
		*/
        virtual void draw_DATA_Labels (
						DataCode dtc, 
						QFont 	 labelsFont,
						QColor   labelsColor,
						QString  (formatLabelFunction) (float v, bool withUnit),
						QPainter &pnt, const Projection *proj);

		/** Pressure: write H and L at hight and low points (pressure).
		*/
        virtual void draw_DATA_MinMax (
						DataCode dtc, 
						double   meanValue,
						QString  minSymbol,
						QString  maxSymbol,
						QFont 	 labelsFont,
						QColor   labelsColor,
						QPainter &pnt, const Projection *proj);
		
		//----------------------------------------------------------------
		// Drawing functions (pure virtual)
		//----------------------------------------------------------------
		/** Draw the complete colored map.
			Update windAltitude.
		*/
		virtual void draw_ColoredMapPlain ( 
						DataCode dtc,
						bool smooth,
						QPainter &pnt, 
						const Projection *proj) = 0;
		
		/** Draw wind arrows on the colored map.
			Update windAltitude.
		*/
		virtual void draw_WIND_Arrows (
						Altitude altitude, 
						bool showBarbules, QColor windArrowsColor, 
						QPainter &pnt, const Projection *proj ) = 0;
		
		/** Draw a mark a each grid vertex.
		*/
        virtual void draw_GridPoints (
						QPainter &pnt, const Projection *proj ) = 0;
        
		//----------------------------------------------------------------
		// Isolines functions
		//----------------------------------------------------------------
		/** Draw all the Isolines contained in a list.
		*/
        virtual void complete_listIsolines (
						std::list<IsoLine *> *listIsolines,
						DataCode dtc,
						double dataMin, double dataMax, double dataStep
						);
						
        void draw_listIsolines (
						std::list<IsoLine *> & listIsolines,
						QPainter &pnt, const Projection *proj);

        void draw_listIsolines_labels (
						std::list<IsoLine *>  & listIsolines,
						double coef,
						QColor &color,
						QPainter &pnt, const Projection *proj);

						
		//----------------------------------------------------------------
		// Drawing auxiliary functions
		//----------------------------------------------------------------
		void draw_CoveredZone (QPainter &pnt, const Projection *proj);

		void drawWindArrowWithBarbs (
        			QPainter &pnt, int i, int j,
        			double vx, double vy,
        			bool south,
        			QColor arrowColor=Qt::white);
		
		//----------------------------------------------------------------
		// Map colors 
		//----------------------------------------------------------------
		void   setCloudsColorMode (QString settingName)
			{ isCloudsColorModeWhite 
				= Util::getSetting(settingName,"white").toString()=="white"; }
		
        QRgb   getAltitudeColor (double v, const Altitude &alt, bool smooth);
        QRgb   getRainColor     (double mm, bool smooth);
        QRgb   getSnowDepthColor(double mm, bool smooth);
        QRgb   getHumidColor    (double v, bool smooth);
        QRgb   getWindColor     (double v, bool smooth);
        QRgb   getTemperatureColor (double v, bool smooth);
        QRgb   getPressureColor    (double v, bool smooth);
		QRgb   getDeltaTemperaturesColor (double v, bool smooth);
        QRgb   getCAPEColor  (double v, bool smooth);
		QRgb   getCloudColor (double v, bool smooth);
		
/*		bool   getMapColorDataMinMaxStep (
					DataCode &dtc, double *min, double *max, double *step);
		QRgb   getMapColorData (DataCode &dtc, double v, bool smooth);*/
		

	protected:
        time_t  	currentDate;
        std::set<time_t>    listDates;     // liste des dates des GribRecord

		bool   mustInterpolateValues;
		bool   drawWindArrowsOnGrid;
		bool   isCloudsColorModeWhite;
		bool	mustDuplicateFirstCumulativeRecord;
        double  isobarsStep;          // Ecart entre isobares
        double  isotherms0Step;       // Ecart entre isothermes 0C

		Altitude windAltitude;		  // current wind altitude
		QColor windArrowColor;        // couleur des flèches du vent
		int    windArrowSpace;        // distance mini entre flèches (pixels)
        int    windArrowSpaceOnGrid;  // distance mini entre flèches si affichage sur grille
        int    windBarbuleSpace;      // distance mini entre flèches (pixels)
        int    windBarbuleSpaceOnGrid;  // distance mini entre flèches
        
        void    drawWindArrow (QPainter &pnt, int i, int j, double vx, double vy);
		void 	initializeColors();

		//-----------------------------------------------------------------
		void  drawColorMapGeneric_1D (
				QPainter &pnt, const Projection *proj, bool smooth,
				DataCode dtc,
				QRgb (GriddedPlotter::*function_getColor) (double v, bool smooth)
			);
		void  drawColorMapGeneric_2D (
				QPainter &pnt, const Projection *proj, bool smooth,
				DataCode dtcX, DataCode dtcY, 
				QRgb (GriddedPlotter::*function_getColor) (double v, bool smooth)
			);
		void  drawColorMapGeneric_Abs_Delta_Data (
				QPainter &pnt, const Projection *proj, bool smooth,
				DataCode dtc1, DataCode dtc2, 
				QRgb (GriddedPlotter::*function_getColor) (double v, bool smooth)
			);
		
		
		
	private:
        QColor windColor[14];        // couleur selon la force du vent en beauforts
        QColor rainColor[17];
        int    mapColorTransp;
		
        int    windArrowSize;         // longueur des flèches
        int    windBarbuleSize;       // longueur des flèches

        void drawTransformedLine( QPainter &pnt,
                double si, double co,int di, int dj, int i,int j, int k,int l);
        
        void drawPetiteBarbule(QPainter &pnt, bool south,
                    double si, double co, int di, int dj, int b);
        void drawGrandeBarbule(QPainter &pnt,  bool south,
                    double si, double co, int di, int dj, int b);
        void drawTriangle(QPainter &pnt, bool south,
                    double si, double co, int di, int dj, int b);
};






#endif
