/*****************************************************************************
 *****************************************************************************
 Copyright (c) 1999-2001, Intel Corporation 

 All rights reserved.

 Redistribution and use in source and binary forms, with or without 
 modification, are permitted provided that the following conditions are met:

 1. Redistributions of source code must retain the above copyright notice, 
 this list of conditions and the following disclaimer.

 2. Redistributions in binary form must reproduce the above copyright notice,
 this list of conditions and the following disclaimer in the documentation 
 and/or other materials provided with the distribution.

 3. Neither the name of Intel Corporation nor the names of its contributors 
 may be used to endorse or promote products derived from this software 
 without specific prior written permission.

 THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 DISCLAIMED. IN NO EVENT SHALL CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT,
 INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF 
 LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
 NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 

 *****************************************************************************
****************************************************************************/
/*
*                                                                     		*
* This software is supplied under the terms of the license included   		*
* above.  All use of this driver must be in accordance with the terms 		*
* of that license.                                                    		*
*                                                                                        *
* Module Name:                                                                           *
*     eeprom.h                                                                           *
*                                                                                        *
* Abstract:                                                                              *
*     This module contains procedure declarations to read and write to a serial EEPROM   *
*                                                                                        *
*     This driver runs on the following hardware:                                        *
*     8255x based PCI 10/100Mb ethernet adapters                                         *
*                                                                                        *
******************************************************************************************/

#ifndef _EEPROM_H_
#define _EEPROM_H_

#include "e100.h"


#define EEPROM_SEMAPHORE_TIMEOUT		500

#define CSR_EEPROM_CONTROL_FIELD(bdp) ((bdp)->bddp->scbp->scb_eprm_cntrl)

#define CSR_GENERAL_CONTROL2_FIELD(bdp) ((bdp)->bddp->scbp->scb_ext.d102_scb.scb_gen_ctrl2)

#define WAIT_FOR_CONDITION(adapter, count, expr)                \
{   unsigned int    counter = (count) / 2;                      \
    while (!(expr) ) {                                          \
        udelay(2);                                              \
        if (!--counter) {                                       \
            break;                                              \
}}}

/*             */

#define STALL_TIME          100
#define EEPROM_CHECKSUM     (__u16)0xBABA


int InitializeEEpromStructure(bd_config_t *adapter);

void InitializeEEpromBuffer(bd_config_t *adapter);
void ShiftOutBits(bd_config_t *adapter, __u16 data, __u16 count);
void RaiseClock(bd_config_t *adapter, __u16 *x);
void LowerClock(bd_config_t *adapter, __u16 *x);
void EEpromCleanup(bd_config_t *adapter);
void UpdateChecksum(bd_config_t *adapter);
void WriteEEprom(bd_config_t *adapter, __u16 reg, __u16 data);
void StandBy(bd_config_t *adapter);

__u16 GetEEpromSize(bd_config_t *adapter);
__u16 ReadEEprom(bd_config_t *adapter, __u16 reg);
__u16 ShiftInBits(bd_config_t *adapter);
__u16 WaitEEpromCmdDone(bd_config_t *adapter);


boolean_t ReadEEpromNodeAddress(bd_config_t *adapter, __u8 *buffer, unsigned int size);


//----------------------------------------------------------------------------------------
// Procedure:   EepromSetSemaphore
//
// Description: This function set (write 1) Gamla EEPROM semaphore bit (bit 23 word 0x1C in the CSR).
//
// Arguments:
//      Adapter                 - Adapter context
//
// Returns:  TRUE if success
//           else return FALSE 
//
//----------------------------------------------------------------------------------------

inline boolean_t EepromSetSemaphore(bd_config_t *adapter)
{
    __u16  data=0;
    
    // Get current value of General Control 2
    data = readb(&CSR_GENERAL_CONTROL2_FIELD(adapter));
    
    // Set bit 23 word 0x1C in the CSR.
    data |= SCB_GCR2_EEPROM_ACCESS_SEMAPHORE;
    writeb(data, &CSR_GENERAL_CONTROL2_FIELD(adapter));

    // Check to see if this bit set or not.
    data = readb(&CSR_GENERAL_CONTROL2_FIELD(adapter));
    
    return ((data & SCB_GCR2_EEPROM_ACCESS_SEMAPHORE) ? B_TRUE: B_FALSE);
}

//----------------------------------------------------------------------------------------
// Procedure:   EepromResetSemaphore
//
// Description: This function reset (write 0) Gamla EEPROM semaphore bit (bit 23 word 0x1C in the CSR).
//
// Arguments:
//      Adapter                 - Adapter context
//
// Returns:  Nothing
//
//----------------------------------------------------------------------------------------

inline void EepromResetSemaphore(bd_config_t *adapter)
{
    __u16  data=0;

    // Get current value of General Control 2
    data = readb(&CSR_GENERAL_CONTROL2_FIELD(adapter));
    
    // Reset bit 23 word 0x1C in the CSR.
    data &= ~(SCB_GCR2_EEPROM_ACCESS_SEMAPHORE) ;
    writeb(data, &CSR_GENERAL_CONTROL2_FIELD(adapter));
}

#endif      // _EEPROM_H_
