/*
    This software may only be used by you under license from AT&T Corp.
    ("AT&T").  A copy of AT&T's Source Code Agreement is available at
    AT&T's Internet website having the URL:
    <http://www.research.att.com/sw/tools/graphviz/license/source.html>
    If you received this software without first entering into a license
    with AT&T, you have an infringing copy of this software and cannot use
    it without violating AT&T's intellectual property rights.
*/
#pragma prototyped

/*
 * Written by Emden R. Gansner
 * Derived from Graham Wills' algorithm described in GD'97.
 */

#include    "circle.h"
#include    "adjust.h"
#include    "neatoprocs.h"

#ifdef DMALLOC
#include "dmalloc.h"
#endif

static void 
twopi_nodesize(node_t* n, boolean flip)
{
  int         w;

  w = n->u.xsize = POINTS(n->u.width);
  n->u.lw  = n->u.rw = w / 2;
  n->u.ht = n->u.ysize = POINTS(n->u.height);
}

static void
twopi_initNode (node_t* n)
{
  char    *str;

  n->u.width = late_float(n,N_width,DEFAULT_NODEWIDTH,MIN_NODEWIDTH);
  n->u.height = late_float(n,N_height,DEFAULT_NODEHEIGHT,MIN_NODEWIDTH);
  if (N_label == NULL) str = n->name;
  else {
    str = agxget(n,N_label->index);
    if (strcmp(str,NODENAME_ESC) == 0) str = n->name;
  }
  n->u.label = make_label(str,
    late_float(n,N_fontsize,DEFAULT_FONTSIZE,MIN_FONTSIZE),
    late_nnstring(n,N_fontname,DEFAULT_FONTNAME),
    late_nnstring(n,N_fontcolor,DEFAULT_COLOR),n->graph);
  n->u.shape = bind_shape(late_nnstring(n,N_shape,DEFAULT_NODESHAPE));
  n->u.shape->initfn(n); /* ### need to quantize ? */
  twopi_nodesize(n,n->graph->u.left_to_right);
  n->u.alg = (void*)zmalloc (sizeof (cdata));
}

static void
twopi_initPort(node_t* n, edge_t* e, char* name)
{
  port_t  port;

  if (name == NULL) return;
  port = n->u.shape->portfn(n,name);
#ifdef NOTDEF
  if (n->graph->u.left_to_right) port.p = invflip_pt(port.p);
#endif
  port.order = 0;
  if (e->tail == n) e->u.tail_port = port; else e->u.head_port = port;
}

static void
twopi_initEdge (edge_t* e)
{
  char    *p;

  e->u.factor = late_float(e,E_weight,1.0,0.0);

  if (E_label && (p = agxget(e,E_label->index)) && (p[0])) {
    e->u.label = make_label(agxget(e,E_label->index),
      late_float(e,E_fontsize,DEFAULT_FONTSIZE,MIN_FONTSIZE),
      late_nnstring(e,E_fontname,DEFAULT_FONTNAME),
      late_nnstring(e,E_fontcolor,DEFAULT_COLOR),e->tail->graph);
  }

  /* vladimir */
  if (E_headlabel && (p = agxget(e,E_headlabel->index)) && (p[0])) {
    e->u.head_label = make_label(agxget(e,E_headlabel->index),
      late_float(e,E_labelfontsize,DEFAULT_FONTSIZE,MIN_FONTSIZE),
      late_nnstring(e,E_labelfontname,DEFAULT_FONTNAME),
      late_nnstring(e,E_labelfontcolor,DEFAULT_COLOR),e->tail->graph);
  }
  if (E_taillabel && (p = agxget(e,E_taillabel->index)) && (p[0])) {
    e->u.tail_label = make_label(agxget(e,E_taillabel->index),
      late_float(e,E_labelfontsize,DEFAULT_FONTSIZE,MIN_FONTSIZE),
      late_nnstring(e,E_labelfontname,DEFAULT_FONTNAME),
      late_nnstring(e,E_labelfontcolor,DEFAULT_COLOR),e->tail->graph);
  }
  /* end vladimir */

  twopi_initPort(e->tail,e,agget(e,"tailport"));
  twopi_initPort(e->head,e,agget(e,"headport"));
}

static void 
twopi_initGraph(graph_t *g)
{
  node_t *n;
  edge_t *e;

  for (n = agfstnode(g); n; n = agnxtnode(g,n)) twopi_initNode(n);
  for (n = agfstnode(g); n; n = agnxtnode(g,n)) {
    for (e = agfstout(g,n); e; e = agnxtout(g,e)) twopi_initEdge(e);
  }
}

void
twopi_layout(Agraph_t* g)
{
  attrsym_t*  sym;
  Agnode_t*   ctr = 0;
  char*       s;

    /* setting rankdir=LR is currently undefined in neato,
     * but having it set causes has effects in the common routines.
     * So, we turn it off.
     */
  sym = agfindattr(g,"rankdir");
  if (sym) agxset (g, sym->index, "");

  graph_init(g);
  g->u.drawing->engine = TWOPI;
  twopi_initGraph(g);
  s = agget (g, "center");
  if (s && (*s != '\0')) {
    ctr = agfindnode (g, s);
    if (!ctr) {
      fprintf (stderr, "Warning: specified center node \"%s\" was not found.",
        s);
      fprintf (stderr, "Using default calculation for center\n");
    }
  }
  circleLayout (g,ctr);
  adjustNodes (g);
  spline_edges(g);
  dotneato_postprocess(g, twopi_nodesize);

}

static void twopi_cleanup_node(node_t* n)
{
	free (n->u.alg);
	if (n->u.shape)
		n->u.shape->freefn(n);
	free_label(n->u.label);
	memset(&(n->u),0,sizeof(Agnodeinfo_t));
}

static void twopi_free_splines(edge_t* e)
{
	int		i;
	if (e->u.spl) {
		for (i = 0; i < e->u.spl->size; i++)
		       	free(e->u.spl->list[i].list);
		free(e->u.spl->list);
		free(e->u.spl);
	}
	e->u.spl = NULL;
}

static void twopi_cleanup_edge(edge_t* e)
{
	twopi_free_splines(e);
	free_label(e->u.label);
	memset(&(e->u),0,sizeof(Agedgeinfo_t));
}

static void twopi_cleanup_graph(graph_t* g)
{
/*	free_scan_graph(g); */
	free_ugraph(g);
	free_label(g->u.label);
	memset(&(g->u),0,sizeof(Agraphinfo_t));
}

void twopi_cleanup(graph_t* g)
{
	node_t  *n;
	edge_t  *e;

	for (n = agfstnode(g); n; n = agnxtnode(g, n)) {
		for (e = agfstedge(g, n); e; e = agnxtedge(g, e, n)) {
			twopi_cleanup_edge(e);
		}
		twopi_cleanup_node(n);
	}
	twopi_cleanup_graph(g);
}
