/*-------------- Telecommunications & Signal Processing Lab ---------------
                             McGill University

Routine:
  int FIreadFilt (const char Fname[], int MaxNcof, float h[], int *Ncof,
		  FILE *fpinfo)

Purpose:
  Read a filter coefficient file

Description:
  This procedure reads filter coefficients from a filter coefficient file.  The
  first line in the file indicates the type of filter.
    !FIR	- FIR filter, direct form
    !IIR	- IIR filter, cascade of biquad sections
    !ALL	- All-pole filter, direct form
    !WIN	- Window coefficients, direct form
    !CAS	- Cascade analog biquad sections
  Subsequent lines contain filter coefficients in text form.  Data fields
  are free format, with data values separated by white-space (as defined by
  isspace).  Zero or more data values can appear in each line of input.  Commas
  can also be used to separate data values, but only within a line, i.e. a
  comma should not appear at the end of a line.  A "!" character marks the
  beginning of a comment that extends to the end of the line.

  This routine prints an error message and halts execution on detection of an
  error.

Parameters:
  <-  int FIreadFilt
      Filter type coded as follows,
        FI_UNDEF = 0, undefined filter file identifier
        FI_FIR   = 1, FIR filter, direct form
        FI_IIR   = 2, IIR filter, cascade of biquad sections
        FI_ALL   = 3, All-pole filter, direct form
        FI_WIN   = 4, Window coefficients, direct form
        FI_CAS   = 5  Cascade analog biquad sections
   -> const char Fname[]
      Filter file name
   -> int MaxNcof
      Maximum number of coefficients to be returned
  <-  float h[]
      Array of Ncof output filter coefficients
  <-  int *Ncof
      Number of filter coefficients returned
   -> FILE *fpinfo
      File pointer for printing filter file information.  If fpinfo is not
      NULL, information about the filter file is printed on the stream
      selected by fpinfo.

Author / revision:
  P. Kabal  Copyright (C) 1997
  $Revision: 1.34 $  $Date: 1997/10/15 21:10:25 $

-------------------------------------------------------------------------*/

static char rcsid[] = "$Id: FIreadFilt.c 1.34 1997/10/15 libtsp-v3r0 $";

#include <libtsp.h>
#include <libtsp/nucleus.h>
#include <libtsp/Xstdio.h>	/* FILENAME_MAX */
#include <libtsp/FIpar.h>
#include <libtsp/FImsg.h>

#define COMMENT_CHAR	'!'

/* Keyword templates for the filter types */
static const char *FItab[] = {
  "!FIR**",	/* FIR filter, direct form */
  "!IIR**",	/* IIR filter, cascade of biquad sections */
  "!ALL**",	/* All-pole filter, direct form */
  "!WIN**",	/* Window coefficients, direct form */
  "!CAS**",	/* Cascade analog biquad sections */
  NULL
};

/* String descriptions of the filter types */
static const char *FItype[] = {
  "Undefined filter type",
  "FIR filter (direct form)",
  "IIR filter (cascade biquad sections)",
  "All-pole filter (direct form)",
  "Window coefficients",
  "Analog filter (Cascade biquad sections)"
};


int
FIreadFilt (Fname, MaxNcof, h, Ncof, fpinfo)

     const char Fname[];
     int MaxNcof;
     float h[];
     int *Ncof;
     FILE *fpinfo;

{
  FILE *fp;
  char *line;
  int N;
  int FiltType;
  char FullName[FILENAME_MAX];

/* Open the filter coefficient file */
  fp = fopen (Fname, "r");
  if (fp == NULL)
    UTerror ("FIreadFilt: %s: \"%s\"", FIM_OpenRErr, Fname);

/* Determine the filter type */
  FiltType = FI_UNDEF;
  line = FLgetLine (fp);
  if (line != NULL)
    FiltType = STkeyMatch (line, FItab) + 1;
  rewind (fp);

/* Decode data records */
  N = FLfReadTF (fp, MaxNcof, COMMENT_CHAR, h);

/* Error checks */
  if (FiltType == FI_IIR) {
    if (N % 5 != 0)
      UThalt ("FIreadFilt: %s", FIM_IIRNCoefErr);
  }

/* Print filter file information */
  if (fpinfo != NULL) {
    FLfullName (Fname, FullName);
    fprintf (fpinfo, FIMF_FiltFile, FullName);
    fprintf (fpinfo, FIMF_FiltType, FItype[FiltType], FLfileDate(fp, 3));
    if (FiltType == FI_IIR) {
      if (N == 5)
	fprintf (fpinfo, FIMF_NumIIR1, N);
      else
	fprintf (fpinfo, FIMF_NumIIRN, N, N / 5);
    }
    else
      fprintf (fpinfo, FIMF_NumCoef, N);
  }

/* Close the file */
  fclose (fp);

/* Return the values */
  *Ncof = N;
  return FiltType;
}
