###############################################################################
##
## Authors: Michel Sanner, Anna Omelchenko
##
##
###############################################################################


import unittest

from DejaVu import Viewer
from DejaVu.Spheres import Spheres
from DejaVu.IndexedPolylines import IndexedPolylines
try:
    from scenario.actor import Actor
except:
    print "Could not import Actor from scenario.actor"
try:
    from scenario.interpolators import FloatScalarInterpolator, FloatVectorInterpolator
except:
   print "Error in importing interpolators from scenario"

try:
    from scenario.datatypes import FloatType, IntType
except:
    print "Error in importing datatypes from scenario"

class foo:
    def __init__(self):
        self.a= 2.0
        
f = foo()

class ActorBaseTest(unittest.TestCase):

    def test_actor_01(self):
        """ test creating an actor with a given initial value"""
        actor = Actor('test', f, 0.5)
        self.assertEqual(len(actor.keyframeValues), 1)
        self.assertEqual(actor.keyframeValues[0], 0.5)
        self.assertEqual(len(actor.keyframes), 1)
        k,v = actor.keyframes.items()[0]
        self.assertEqual(k, 0)
        self.assertEqual(v, 0)
                         
    def test_actor_02(self):
        """test creating an actor with a given interpolator and initial value """
        actor = Actor('test', f, 0.5, FloatType, FloatScalarInterpolator)
        inter, k1, k2, valGen = actor.getInterval(0)
        self.assertEqual(inter,0)
        self.assertEqual(k1, 0)
        self.assertEqual(k2, None)
        
        # check that frame 10 is in interval
        inter, k1, k2, valGen = actor.getInterval(10)
        self.assertEqual(inter, 0)
        self.assertEqual(k1, 0)
        self.assertEqual(k2, None)
        
        # check the values we get at 0
        val, inter, valGen = actor.getValueAt(0)
        self.assertEqual(val, 0.5)
        self.assertEqual(inter, -1)
        self.assertEqual(valGen, -1)

    def test_actor_03(self):
        """test creating an actor with a given interpolator, initial value and wrong value type"""
        
        actor = Actor('test', f, 0.5, IntType, FloatScalarInterpolator)
        vals = actor.keyframeValues
        self.assertEqual(vals, [0.5])
        actor.setKeyframe( 10, 10.5)
        #should get a warning here. The keyframes of the actor shoud not get modified:
        vals = actor.keyframeValues
        self.assertEqual(vals, [0.5])
        
    def test_actor_04(self):
        """test adding a keyframe to an actor"""
        actor = Actor('test', f, 0.5, FloatType, FloatScalarInterpolator)
        actor.setKeyframe( 10, 10.5)
        self.assertEqual( len(actor.keyframeValues), 2)
        self.assertEqual( actor.keyframeValues[1], 10.5)
        
        # make sure the actor's last frame is correct
        self.assertEqual( actor.getLastFrame(), 10)
        
        # check the keyframe positions
        self.assertEqual( actor.keyframes[0], 0)
        self.assertEqual( actor.keyframes[10], 1)
        
        # check the starting and ending values of the interpolators
        self.assertEqual( actor.valueGenerators[0].firstVal, 0.5)
        self.assertEqual( actor.valueGenerators[0].lastVal, 10.5)
        self.assertEqual( actor.valueGenerators[1].firstVal, 10.5)
        self.assertEqual( actor.valueGenerators[1].lastVal, 10.5)
        
        # check that we get the right interval for 0, 5 and 15
        inter, k1, k2, valGen = actor.getInterval(0)
        self.assertEqual( inter, 0)
        self.assertEqual( k1, 0)
        self.assertEqual( k2, 10)
        self.assertEqual( valGen, actor.valueGenerators[0])
        
        inter, k1, k2, valGen = actor.getInterval(5)
        self.assertEqual( inter, 0)
        self.assertEqual( k1, 0)
        self.assertEqual( k2, 10)
        self.assertEqual( valGen, actor.valueGenerators[0])
        
        # at 10 we should get the second interval
        inter, k1, k2, valGen = actor.getInterval(10)
        self.assertEqual( inter, 1)
        self.assertEqual( k1, 10)
        self.assertEqual( k2, None)
        self.assertEqual( valGen, actor.valueGenerators[1])
        
        inter, k1, k2, valGen = actor.getInterval(15)
        self.assertEqual( inter, 1)
        self.assertEqual( k1, 10)
        self.assertEqual( k2, None)
        self.assertEqual( valGen, actor.valueGenerators[1])
         
         
    def test_actor_05(self):
        """test keyframe values of an actor"""
        actor = Actor('test', f, 0.5, FloatType, FloatScalarInterpolator)
        actor.setKeyframe( 10, 10.5)
        val, inter, valGen = actor.getValueAt(0)
        self.assertEqual( val, 0.5)
        self.assertEqual( inter, -1) # we are on a keyframe
        self.assertEqual( valGen, -1) 
        
        val, inter, valGen = actor.getValueAt(10)
        self.assertEqual( val, 10.5)
        self.assertEqual( inter, -1)
        self.assertEqual( valGen, -1)
        
        val, inter, valGen = actor.getValueAt(5)
        self.assertEqual( val, 5.5)
        self.assertEqual( inter, 0)
        self.assertEqual( valGen, actor.valueGenerators[0])

    def test_actor_06(self):
        """ test adding two keyframes (at 10.5 and 20)"""
        actor = Actor('test', f, 0.5, FloatType, FloatScalarInterpolator)
        actor.setKeyframe( 10, 10.5)
        actor.setKeyframe(20, 15.0)
        val, inter, valGen = actor.getValueAt(0)
        self.assertEqual( val, 0.5)
        self.assertEqual( inter, -1) # we are on a keyframe
        self.assertEqual( valGen, -1) 
        
        val, inter, valGen = actor.getValueAt(10)
        self.assertEqual( val, 10.5)
        self.assertEqual( inter, -1)
        self.assertEqual( valGen, -1)
        
        val, inter, valGen = actor.getValueAt(5)
        self.assertEqual( val, 5.5)
        self.assertEqual( inter, 0)
        self.assertEqual( valGen, actor.valueGenerators[0])

        val, inter, valGen = actor.getValueAt(20)
        self.assertEqual( val, 15.0)
        self.assertEqual( inter, -1)
        self.assertEqual( valGen, -1)
        
        val, inter, valGen = actor.getValueAt(15)
        self.assertEqual( val, 12.75)
        self.assertEqual( inter, 1)
        self.assertEqual( valGen, actor.valueGenerators[1])
        
    def test_actor_07(self):
        """test inserting a keyframe between two existing keyframes """
        actor = Actor('test', f, 0.5, FloatType, FloatScalarInterpolator)
        actor.setKeyframe( 20, 15.0)
        actor.setKeyframe( 10, 10.5)
        self.assertEqual(actor.keyframes, {0: 0, 10: 2, 20: 1})
        val, inter, valGen = actor.getValueAt(0)
        self.assertEqual( val, 0.5)
        self.assertEqual( inter, -1) # we are on a keyframe
        self.assertEqual( valGen, -1) 
        
        val, inter, valGen = actor.getValueAt(10)
        self.assertEqual( val, 10.5)
        self.assertEqual( inter, -1)
        self.assertEqual( valGen, -1)
        
        val, inter, valGen = actor.getValueAt(5)
        self.assertEqual( val, 5.5)
        self.assertEqual( inter, 0)
        self.assertEqual( valGen, actor.valueGenerators[0])

        val, inter, valGen = actor.getValueAt(20)
        self.assertEqual( val, 15.0)
        self.assertEqual( inter, -1)
        self.assertEqual( valGen, -1)
        
        val, inter, valGen = actor.getValueAt(15)
        self.assertEqual( val, 12.75)
        self.assertEqual( inter, 1)
        self.assertEqual( valGen, actor.valueGenerators[1])
        

    def test_actor_08(self):
        """testing calling with default args"""
        vi = Viewer()
        sph = Spheres('spheres', vertices =((0,0,0),(5.,0.,0.)),
                  materials=((1,0,0),),
                  radii=2.0, quality=30, inheritMaterial=False)
        vi.AddObject(sph)
        actor=Actor("spheres_radii",sph, 2.0,  FloatType, FloatScalarInterpolator)
        actor.setKeyframe( 50, 6.0)
        for i in range(51):
            val, interp, valGen = actor.getValueAt(i)
            sph.Set(radii = val)
        self.assertEqual(sph.radius, 6.0)
        l = range(51)
        l.reverse()
        for i in l:
            val, interp, valGen = actor.getValueAt(i)
            sph.Set(radii = val)
        self.assertEqual(sph.radius, 2.0)
        vi.Exit()

    def test_actor_09(self):
        """test setting a different interpolator (value generator) for an interval"""
        
        vi = Viewer()
        sph = Spheres('spheres', vertices =((0,0,0),(5.,0.,0.)),
                  materials=((1,0,0),),
                  radii=2.0, quality=30, inheritMaterial=False)
        vi.AddObject(sph)
        actor=Actor("spheres_radii",sph, 2.0,  FloatType, FloatScalarInterpolator)
        actor.setKeyframe(40, 3.0)
        actor.setKeyframe(70, 6.0)
        actor.setKeyframe(100, 10.0)
        interp = FloatVectorInterpolator((3.0, 1.0), (6.0, 4.0))
        actor.setValueGenerator(interp, 1)
        #print actor.valueGenerators
        for i in range(41):
            val, interp, valGen = actor.getValueAt(i)
            sph.Set(radii = val)
        
        print "at frame", i, val
        self.assertEqual(val, 3.0)
        for i in range(41, 70):
            val, interp, valGen = actor.getValueAt(i)
            sph.Set(radii = val)
        print "at frame ",i,  val
        self.assertEqual(len(val), 2)
        for i in range(70,101):
            val, interp, valGen = actor.getValueAt(i)
            sph.Set(radii = val)
        print "at frame", i, val
        self.assertEqual(val, 10.0)
        vi.Exit()

    
if __name__ == '__main__':
    unittest.main()
