/*
 *   dspload.c -- DSP Load module/task/seg/etc. routines 
 *
 *  Written By: Mike Sullivan IBM Corporation
 *
 *  Copyright (C) 1999 IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify      
 * it under the terms of the GNU General Public License as published by      
 * the Free Software Foundation; either version 2 of the License, or         
 * (at your option) any later version.                                       
 *                                                                           
 * This program is distributed in the hope that it will be useful,           
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             
 * GNU General Public License for more details.                              
 *                                                                           
 * NO WARRANTY                                                               
 * THE PROGRAM IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OR        
 * CONDITIONS OF ANY KIND, EITHER EXPRESS OR IMPLIED INCLUDING, WITHOUT      
 * LIMITATION, ANY WARRANTIES OR CONDITIONS OF TITLE, NON-INFRINGEMENT,      
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. Each Recipient is    
 * solely responsible for determining the appropriateness of using and       
 * distributing the Program and assumes all risks associated with its        
 * exercise of rights under this Agreement, including but not limited to     
 * the risks and costs of program errors, damage to or loss of data,         
 * programs or equipment, and unavailability or interruption of operations.  
 *                                                                           
 * DISCLAIMER OF LIABILITY                                                   
 * NEITHER RECIPIENT NOR ANY CONTRIBUTORS SHALL HAVE ANY LIABILITY FOR ANY   
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL        
 * DAMAGES (INCLUDING WITHOUT LIMITATION LOST PROFITS), HOWEVER CAUSED AND   
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR     
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE    
 * USE OR DISTRIBUTION OF THE PROGRAM OR THE EXERCISE OF ANY RIGHTS GRANTED  
 * HEREUNDER, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGES             
 *                                                                           
 * You should have received a copy of the GNU General Public License         
 * along with this program; if not, write to the Free Software               
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 *                                                                           
 * 
 *  10/23/2000 - Alpha Release 0.1.0
 *            First release to the public
 *
 */

#include <stdio.h>
#include <string.h>
#include <port_types.h>
#include "dspmgr.h"

#include "dspio.h"                     // Include IO routines
#include "dspload.h"                   // Include function prototypes
#include "dspparse.h"                  // Include function prototypes
#include "dspalloc.h"                  // Include function prototypes
#include "dspactiv.h"                  // Include function prototypes
#include "dspbios.h"                   // Phase V
#include "dspfixup.h"                  // Include function prototypes
#include "dspfree.h"                   // Include function prototypes
#include "dspglist.h"                  // Include function prototypes
#include "dspgpc.h"                    // Include function prototypes
#include "dspstruc.h"                  // Include function prototypes
#include "dsppcmem.h"                  // for Allocate Perm
#include "dspipc.h"                    //
#include "dspstcmp.h"                  // Include function prototypes
#include "dspquery.h"                  // CH01 Include NameToXXX functions

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: LoadModule                                              */
/*                                                                          */
/* FUNCTION: This routine loads a module into the DSP.                      */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRDSP   prdsp       - Load the module onto this DSP.              */
/*        PSZ     pszName     - Name of the module to load.                 */
/*        PSZ     pszFileName - Name of the file to load the module from.   */
/*        USHORT  usLoadFlags - Start the module running in this activation */
/*                              state and also the priority of module       */
/*        PRMOD * pprmod      - Ptr to module ptr.(return info)             */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        PRMOD * pprmod      - Update To point to new module structure.    */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC LoadModule(PRDSP prdsp,PSZ pszFileName,PSZ pszName,USHORT usLoadFlags,PRMOD
               *pprmod)

{
   PRTSK      prtsk;                   // Current task node pointer
   PRTSK      prtskHead;               // Head of task list
   PRMOD      prmod = NULL;            // Ptr to module; code assumes NULL
   PRMOD      prmodTemp;               // Ptr to module Tail
   HFILE      hfile;                   // Handle to filename
   RC         ulRC = DSP_NOERROR;      // Assume DSP_NOERROR
   USHORT     usrcFlags = DSP_NOERROR; // Assume DSP_NOERROR


   MW_SYSLOG_5(TRACE_MANAGER_CORE,"dspload::LoadModule entry prdsp %x pszFileName %s pszName %s usLoadFlags %x\n",
	   (int)prdsp,pszFileName,pszName,(int)usLoadFlags);
   ulRC = CheckDSPStatus(prdsp);       /* CH02 */
   if (ulRC != DSP_NOERROR)
      return (ulRC);

   /*************************************************************************/
   /*  Test to see if the module is already loaded.                         */
   /*************************************************************************/

   if ((prdsp->DSP_prmgr->MGR_prmodTail != NULL) && (pszName != NULL)) {
      prmodTemp = prdsp->DSP_prmgr->MGR_prmodTail;
      do {
         if (prmodTemp->MOD_pszName != NULL) {
            if (mwstrcmp(prmodTemp->MOD_pszName, pszName) == 0) {
	      MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspload::LoadModule error ulRC %x Name: %s\n", DSP_DUPLICATE_NAME,pszName);
               return (DSP_DUPLICATE_NAME);
            }
         }
         prmodTemp = prmodTemp->MOD_prmodNext;
      }  while (prmodTemp != prdsp->DSP_prmgr->MGR_prmodTail);
   }

   /*************************************************************************/
   /* If the filename is NULL, a Dummy Module is created.  Note that the    */
   /* checking above does the verification of a non-duplicate name.  We     */
   /* still have to create a name field if there is a name.  Finally, have  */
   /* to give the module the input state.                                   */
   /*************************************************************************/

   if (pszFileName == NULL) {

      ulRC = AllocRModule(&prmod);

      if (ulRC == DSP_NOERROR) {

         pg->prmodCurrentMod = prmod;
         if (pszName != NULL) {
            ulRC = GetStr(pszName, &(prmod->MOD_pszName));
         }
         else
            prmod->MOD_pszName = pszName;

         if (ulRC == DSP_NOERROR) {

      /* Attach the module data tree structure to the manager module list Tail*/

            CLLInsTail((PVOID)prmod, (PVOID *)&prmod->MOD_prmodNext, (PVOID *)
               &prdsp->DSP_prmgr->MGR_prmodTail->MOD_prmodNext, (PVOID *)
               &prdsp->DSP_prmgr->MGR_prmodTail);

            prmod->MOD_usflgState = usLoadFlags;
         }                             /* endif                             */
      }                                /* endif                             */
   }
   else {

      if ((ulRC = DspOpenFile(pszFileName, &hfile,
                              DSPOPENBINARY|DSPOPENRDONLY)) == DSP_NOERROR) {

             /* parse the module from the .dsp file                         */
             /* If pszName is NULL, ParseModule  Handles.                   */

         if ((ulRC = ParseModule(hfile, pszName, &prmod)) == DSP_NOERROR) {

                /* Setup ptr in DSP structure to ISPOS labels (module ptr)  */
                /* CharliePatch...only set DSP_prmodOS on OS load!          */

            if (fDSPINIT_FLAG)
               prdsp->DSP_prmodOS = prmod;

      /* Attach the module data tree structure to the manager module list Tail*/

            CLLInsTail((PVOID)prmod, (PVOID *)&prmod->MOD_prmodNext, (PVOID *)
               &prdsp->DSP_prmgr->MGR_prmodTail->MOD_prmodNext, (PVOID *)
               &prdsp->DSP_prmgr->MGR_prmodTail);

             /* Traverse the module (each task) and point each task to the DSP*/

            prtskHead = prmod->MOD_prtskTail->TSK_prtskNext; // set task head
            prtsk = prtskHead;         // Start at head of task list

            do {
               prtsk->TSK_prdsp = prdsp; // point task to DSP
               prtsk = prtsk->TSK_prtskNext; // go to next in list
            }  while (prtsk != prtskHead); // Back at head yet ?

                /* Allocate dsp resources for the module                    */

            if ((ulRC = AllocateModule(prmod)) == DSP_NOERROR) {

                   /* Do fixup for the labels in the module                 */

               if ((ulRC = FixupModule(prmod)) == DSP_NOERROR) {

                  /* Set priority of module                                 */
                  /**********************************************************/
                  /* Check activation flag from load request. If inactive   */
                  /* flag is set then stop, module is currently in inactive */
                  /* state. Else, check standby flag, if set then activate  */
                  /* the module into standby state.                         */
                  /**********************************************************/

                  if (((usLoadFlags&DSP_ACTIVATE_MASK) == DSP_ACTIVATE_STANDBY
                     ) || ((usLoadFlags&DSP_ACTIVATE_MASK) ==
                     DSP_ACTIVATE_ACTIVE))
                     ulRC = ChangeModuleState(prmod, (USHORT)(usLoadFlags
                        &(USHORT)DSP_ACTIVATE_MASK));
               }

            }                          /* AllocateModule                    */
         }
         else
            usrcFlags = DSP_LOAD_FAILED;/* Parse Failed                     */
      }                                /* Open File                         */
      else
         return (ulRC);                /* if not, return the error          */

      DspCloseFile(hfile);
   }                                   /* endif FILENAME!=NULL              */
   if (ulRC != DSP_NOERROR) {
     MW_SYSLOG_2(TRACE_MANAGER_CORE,"dspload::LoadModule error LoadModule failed: %s\n", pszFileName);
      /* Error occurred, Unload the module and free all data structures     */
      FreeModule(prmod, usrcFlags);
   }
   else {
      *pprmod = prmod;

   }
   MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspload::LoadModule exit ulRC %lx pprmod %x\n",ulRC,(int)*pprmod);
   return (ulRC);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: LoadTask                                                */
/*                                                                          */
/* FUNCTION: This routine loads a task into the DSP.                        */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRMOD   prmod       - DSP module to place the task under.         */
/*        PSZ     pszFileName - Name of the file to load the task from.     */
/*        PSZ     pszRealName - Name of task in the DSP file to be loaded.  */
/*        PSZ     pszVirtName - Name of task after loading.                 */
/*        USHORT  usLoadFlags - Start the task running in this activation   */
/*                              state, if no flag is sepcified then         */
/*                              DSP_LOAD_STANDBY is implied                 */
/*        PRTSK * pprtsk      - Ptr to task ptr.(return info)               */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        PRTSK * pprtsk      - Update To point to new task structure.      */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC LoadTask(PRMOD prmod,PRDSP prdsp,PSZ pszFileName,PSZ pszRealName,PSZ
             pszVirtName,USHORT usLoadFlags,PVOID pRqHWBuf,PRTSK *pprtsk)

{
   PRTSK      prtsk = NULL;            // Ptr to task; code assumes NULL
   PRTSK      prtskTemp;               // ptr to task temp
   PRTSK      prtskHead;               // head of task list
   HFILE      hfile;                   // Handle to filename
   RC         ulRC;

   MW_SYSLOG_7(TRACE_MANAGER_CORE,"dspload::LoadTask entry prmod %x prdsp %x pszFileName %s pszRealName %s pszVirtName %s usLoadFlags %x\n",
	   (int)prmod,(int)prdsp,pszFileName,pszRealName,pszVirtName,usLoadFlags);
   if (prmod->MOD_achValid != MODVALIDATE)
      return (DSP_INV_HANDLE);

   ulRC = CheckDSPStatus(prdsp);       /* CH01 */
   if (ulRC != DSP_NOERROR)
      return (ulRC);

   if (pRqHWBuf != NULL)
      return (DSP_INV_PARAMETER);

   /*************************************************************************/
   /*  Test to see if the task name is already on module.                   */
   /*************************************************************************/

   if ((prmod->MOD_prtskTail != NULL) && (pszVirtName != NULL)) {
      prtskTemp = prmod->MOD_prtskTail;
      do {
         if (prtskTemp->TSK_pszVirtName != NULL)
            {
            if (mwstrcmp(prtskTemp->TSK_pszVirtName, pszVirtName) == 0) {
	      MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspload::LoadTask error ulRC %x Name: %s\n", DSP_DUPLICATE_NAME,pszVirtName);
               return (DSP_DUPLICATE_NAME);
            }
         }
         prtskTemp = prtskTemp->TSK_prtskNext;
      }  while (prtskTemp != prmod->MOD_prtskTail);
   }
   if ((ulRC = DspOpenFile(pszFileName, &hfile, DSPOPENBINARY|DSPOPENRDONLY))
        == DSP_NOERROR) {

      /**********************************************************************/
      /* parse the task from the .dsp file, this also allocate memory       */
      /* for the task in perm memory.                                       */
      /* ParseTask will have to allow NULL Names.                           */
      /**********************************************************************/

      if ((ulRC = ParseTask(hfile, pszRealName, pszVirtName, prmod, &prtsk))
         == DSP_NOERROR)
         {

         /*******************************************************************/
         /* Use DSP handle provided in API.                                 */
         /*******************************************************************/

         if ((prtsk->TSK_prdsp == NULL) && (prdsp != NULL)) {
            prtsk->TSK_prdsp = prdsp;
         }
         else {

                /************************************************************/
                /* Traverse the module (each task), if find a valid DSP,    */
                /* assign that DSP to the parent DSP of the new task, if not*/
                /* Error.                                                   */
                /************************************************************/
                /*     Note that this is only valid for single DSP          */
                /*     sytems and that we are assuming a Tasks are          */
                /*     on Same DSP.                                         */
                /************************************************************/

            prtskHead = prmod->MOD_prtskTail->TSK_prtskNext; // set task head
            prtskTemp = prtskHead;     // Start at head of task list

            do {
               if (prtskTemp->TSK_prdsp != NULL) { // valid DSP ?
                  prtsk->TSK_prdsp = prtskTemp->TSK_prdsp; // assign DSP to new task
                  break;
               }
               prtskTemp = prtskTemp->TSK_prtskNext; // go to next in list
            }  while (prtskTemp != prtskHead); // Back at head yet ?

                /************************************************************/
                /*     If the Task still does not have a PRDSP              */
                /*     and the pRqHWBuf does not specify a DSP              */
                /*     Default to the first DSP on the Manager.             */
                /* ELSE Temporary for Single DSP systems.                   */
                /*      assign Task to DSP specified in pRqHWBuf            */
                /************************************************************/

            if ((prtsk->TSK_prdsp == NULL) && (prdsp == NULL))
               {
               prtsk->TSK_prdsp = pg->prmgrTail->MGR_prdspTail;
            }
         }                             /* end else prdsp==NULL              */

         /* Allocate dsp resources for the task                             */

         if (ulRC == DSP_NOERROR) {
            if ((ulRC = AllocateTask(prtsk)) == DSP_NOERROR) {

              /* Do fixup for the labels in the Task                        */

               if ((ulRC = FixupTask(prtsk)) == DSP_NOERROR) {

                  /* Set priority of task                                   */
                  /**********************************************************/
                  /* Check activation flag from load request. If inactive   */
                  /* flag is set then stop, task is currently in inactive   */
                  /* state. Else, check standby flag, if set then activate  */
                  /* the task into standby state.  Allow ACTIVE State Also. */
                  /**********************************************************/

                  if (((usLoadFlags&DSP_ACTIVATE_MASK) == DSP_ACTIVATE_STANDBY
                     ) || ((usLoadFlags&DSP_ACTIVATE_MASK) ==
                     DSP_ACTIVATE_ACTIVE))
                     ulRC = ChangeTaskState(prtsk, (USHORT)(usLoadFlags
                        &(USHORT)DSP_ACTIVATE_MASK));
               }
            }
         }                             /* endif DSPNOERROR Check            */
      }                                /* End ParseTask                     */
   }
   else
      return (ulRC);                   /* if not, return the error          */

   DspCloseFile(hfile);

   if (ulRC != DSP_NOERROR) {
      /* Error occurred,Unload the task and free all data structures*/
      FreeTask(prtsk);
   }
   else {
      *pprtsk = prtsk;

   }
   MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspload::LoadTask exit ulRC %lx pprtsk %x\n",ulRC,(int)*pprtsk);
   return (ulRC);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: LoadSegment                                             */
/*                                                                          */
/* FUNCTION: This routine loads a segment into the DSP.                     */
/*                                                                          */
/* INPUT:                                                                   */
/*        PRTSK   prtsk       - DSP task to place the segment under.        */
/*        PSZ     pszFileName - Name of the file to load the segment from.  */
/*        PSZ     pszRealName - Name of seg in the DSP file to be loaded.   */
/*        PSZ     pszVirtName - Name of seg after loading.                  */
/*        HSEG    hSegAddr    - Handle to the segment that stores the       */
/*                              loaded DSP segment address. If hSegAddr     */
/*                              is NULL, no DSP address will be stored.     */
/*        ULONG   ulOffset    - Offset to location that stores the loaded   */
/*                              segment address.                            */
/*        PRSEG * pprseg      - Ptr to segment ptr.(return info)            */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        PRSEG * pprseg      - Update To point to new segment structure.   */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC LoadSegment(PRTSK prtsk,PSZ pszFileName,PSZ pszRealName,PSZ pszVirtName,
                ULONG ulOffset,PRSEG *pprseg)

{

   HFILE      hfile;                   // Handle to filename
   HSEG       prseg = NULL;            // Code assumes initial NULL pointer
   PRGPC      prgpc;
   PRLDMA     prldmaNode;
   RC         ulRC;                    // Error return code
   ULONG      ulsize;
   ULONG      ulDummy;
   LONG       lDummy;
   USHORT     usTemp;                  /* CH01 needed for AIX compatibility */
   BOOL       bStrideHold = FALSE;     // Assume FALSE
   PRSEG      prsegTemp;               // CH02

   MW_SYSLOG_6(TRACE_MANAGER_CORE,"dspload::LoadSegment entry prtsk %x pszFileName %s pszRealName %s pszVirtName %s ulOffset %lx\n",
	   (int)prtsk,pszFileName,pszRealName,pszVirtName,ulOffset);

   if (prtsk->TSK_achValid != TSKVALIDATE)
      return (DSP_INV_HANDLE);

   if ((ulRC=CheckDSPStatus(prtsk->TSK_prdsp)))
      return (ulRC);  /* CH03 Check for card unplugged */

   if ((ulRC = DspOpenFile(pszFileName, &hfile, DSPOPENBINARY|DSPOPENRDONLY))
        == DSP_NOERROR) {

      /**********************************************************************/
      /* Check for duplicate instance names for this segment    CH02        */
      /**********************************************************************/
      if (pszVirtName != NULL) {        // if non null instance name
         /*First check instruction segments                                 */
         if (prtsk->TSK_prsegITail != NULL) {   // if existing instruction segs
            prsegTemp=prtsk->TSK_prsegITail;    // set ptr to segment at tail
            do {                        // thread through Iseg list
               if (prsegTemp->SEG_pszVirtName != NULL) { // if this one has a name
                  if (mwstrcmp(prsegTemp->SEG_pszVirtName,
                        pszVirtName) == 0)      // if duplicate
                  {
		    MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspload::LoadSegment error ulRC %x Code Segment: %s\n",DSP_DUPLICATE_NAME,
                               pszVirtName);  
                     DspCloseFile(hfile);    /* close the file */
                     return (DSP_DUPLICATE_NAME);
                  }
               }
               prsegTemp=prsegTemp->SEG_prsegNext;      // bump to next seg
            } while (prsegTemp != prtsk->TSK_prsegITail);
         }

         /*Now check data segments                                          */
         if (prtsk->TSK_prsegDTail != NULL) {   // if existing data segs
            prsegTemp=prtsk->TSK_prsegDTail;    // set ptr to segment at tail
            do {                        // thread through Dseg list
               if (prsegTemp->SEG_pszVirtName != NULL) { // if this one has a name
                  if (mwstrcmp(prsegTemp->SEG_pszVirtName,
                        pszVirtName) == 0)      // if duplicate
                  {
		    MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspload::LoadSegment error ulRC %x Data Segment: %s\n",DSP_DUPLICATE_NAME,
                               pszVirtName);  
                     DspCloseFile(hfile);     /* close the file */
                     return (DSP_DUPLICATE_NAME);
                  }
               }
               prsegTemp=prsegTemp->SEG_prsegNext;      // bump to next seg
            } while (prsegTemp != prtsk->TSK_prsegDTail);
         }
      }
      /*******************end CH02 ******************************************/

      /**********************************************************************/
      /* parse the Seg from the .dsp file, this also allocate memory        */
      /* for the segment in perm memory.                                    */
      /**********************************************************************/

      if ((ulRC = ParseSeg(hfile, pszRealName, prtsk, &prseg,
         FIND_SEGMENT_OFFSET)) == DSP_NOERROR) {

      /**********************************************************************/
      /* Switch the Segments name to the Virtual Name if supplied.  At this */
      /* point the RSEG structure has two pointers both pointing to the     */
      /* same psz buffer (real & virt).  If the caller has a virtname then  */
      /* we must create a new buffer and copy the virtname to it otherwise  */
      /* we must make the virtname pointer null.  The space for both of     */
      /* these PSZs is freed in RemoveRSeg                                  */
      /**********************************************************************/

         if (pszVirtName != NULL) {

            ulsize = strlen(pszVirtName)+1;
            if ((ulRC = AllocatePerm(ulsize, (PVOID)&prseg->SEG_pszVirtName))
                                                               != DSP_NOERROR)
               {
               DspCloseFile(hfile);     /*CH02*/      /* close the file */
               return (ulRC);
            }

            memcpy(prseg->SEG_pszVirtName, pszVirtName, ulsize);

         }
         else
            prseg->SEG_pszVirtName = NULL;

         /* Allocate dsp resources for the segment                          */

         if ((ulRC = AllocSegRes(prseg)) == DSP_NOERROR) {

            /****************************************************************/
            /* Allocate Resources that would normally be                    */
            /* allocated for an entire task for the segment.                */
            /* GPC, MIPS, IPC                                               */
            /*                                                              */
            /* Create the GPC buffer For Each Owner GPC hanging off GPC list*/
            /* The GPC control blocks should only be hanging off a Task.    */
            /* The GPC buffer will be allocate from DSP memory.             */
            /****************************************************************/

            if ((prgpc = prseg->SEG_prtsk->TSK_prgpcTail) != NULL) {
               do {
                  prgpc = prgpc->GPC_prgpcNext;/* Get next GPC block        */
                  if (prgpc->GPC_prseg == prseg) {
                     if (((prgpc->GPC_prseg->SEG_usflg&SEG_TYPEMASK) ==
                        SEG_DATA) && (prgpc->GPC_usflgKind == GPC_OWNER)) {

                        if ((ulRC = AllocateGPCBuf(prgpc)) != DSP_NOERROR) {
                           DspCloseFile(hfile); /*CH02*/  /* close the file */
                           return  ulRC;
                        }
                     }
                  }

               }  while (prgpc != prseg->SEG_prtsk->TSK_prgpcTail);
            }                          /* endif                             */

            /****************************************************************/
            /*  Allocate DMA for dspLoadedSegments                          */
            /****************************************************************/

            if (prtsk->TSK_prldmaTail != NULL)/* any dma requests for parent*/
                                       /* task                              */
               {
               ulRC = AllocateDMA(&lDummy, &ulDummy);/* allocate dma        */
               if (ulRC != DSP_NOERROR) {                /* if alloc failed */
                  FreeSegment(prseg);  /* free the segment                  */
                  DspCloseFile(hfile); /* close the file                    */
                  return (ulRC);
               }

               /*************************************************************/
               /* See if any stride/hold usage requested                    */
               /*************************************************************/

               prldmaNode = prtsk->TSK_prldmaTail;
               do                      /* thread thru all dmas for parent   */
                                       /* task                              */
                  {
                  if (prldmaNode->LDMA_ulStride != 0)/* find a stride entry?*/
                     {
                     bStrideHold = TRUE;/* indicate at least 1 stride found */
                     break;
                  }
               }  while (prldmaNode != prtsk->TSK_prldmaTail);/* still not  */
                                       /* end                               */
               if (bStrideHold == TRUE)/* any stride/hold requested?        */
                  {
                  ulRC = AllocateStrideHold(prtsk);/* alloc stride/hold     */
                  if (ulRC != DSP_NOERROR)/* if alloc failed                */
                     {
                     FreeSegment(prseg);/* free segment                     */
                     DspCloseFile(hfile);/* close the file                  */
                     return (ulRC);
                  }
               }
            }

            /****************************************************************/
            /*  Allocate IPC for dspLoadedSegments                          */
            /****************************************************************/

            if (prtsk->TSK_prseg_IPC != NULL)/* any ipc requests for parent */
                                       /* task                              */

               /*is ipc associated with this newly loaded segment?          */

               if (prtsk->TSK_prseg_IPC == prseg) {
                  ulRC = AddIPC(prtsk);
                  if (ulRC != DSP_NOERROR) {
                     FreeSegment(prseg);         /* free segment            */
                     DspCloseFile(hfile);        /* close the file          */
                     return (ulRC);
                  }
               }

            /****************************************************************/
            /* Fixup the segment.                                           */
            /****************************************************************/

            ulRC = FixupSegment(prseg);

            /****************************************************************/
            /* Store the loaded DSP address in the segment                  */
            /* handle and offset that are given                             */
            /****************************************************************/

            if ((ulRC == DSP_NOERROR) && (prseg->SEG_prmem != NULL)) {/*    */

            /* stop if empty segment (NULL ptr)                             */
               /* CH01 DS Writes of size 1 must use a USHORT */
               usTemp = (USHORT) prseg->SEG_prmem->MEM_ulSegAddr; /* CH01 */

               PRINT_DEBUG_FOPEN(EMBEDFILEPTR, EMBEDFILENAME, "a",
                    "dspload::LoadSegment");
               PRINT_DEBUG(EMBEDFILEPTR, "  prtsk (0x%X)", prtsk);
               PRINT_DEBUG(EMBEDFILEPTR, "  pszFileName \"%s\"", pszFileName);
               PRINT_DEBUG(EMBEDFILEPTR, "  pszRealName \"%s\"\n",
                     pszRealName);
               PRINT_DEBUG(EMBEDFILEPTR, "  pszVirtName \"%s\"", pszVirtName);
               PRINT_DEBUG(EMBEDFILEPTR, "  ulOffset (0x%X)\n", ulOffset);
               PRINT_DEBUG_BIOS_WRITE(EMBEDFILEPTR,
                    "WRITE21: loaded DSP addr in given seg handle and offset",
                    EOF, 01L,
                    prtsk->TSK_prdsp, ulOffset, 01L, (PVOID) &usTemp);

               ulRC = DspBIOS_D_Write(prtsk->TSK_prdsp, ulOffset, 01L, (PVOID)
                  &usTemp);  /* CH01 */
            }
         }
      }
   }
   else
      return (ulRC);                   /* if not, return the error          */

   DspCloseFile(hfile);

   if (ulRC != DSP_NOERROR) {
      /* Error occurred,Unload the segment and free all data structures*/
      FreeSegment(prseg);
   }
   else {
      *pprseg = prseg;
   }

   MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspload::LoadSegment exit ulRC %lx pprseg %x\n",ulRC,(int)*pprseg);

   return (ulRC);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: ConnectorToBIOS                                         */
/*                                                                          */
/* FUNCTION: This routine takes a connector name and searches the BIOS      */
/*           table for the name.                                            */
/*                                                                          */
/* INPUT:                                                                   */
/*        PSZ         pszConnector - Connector name.                        */
/*        PRMGR   prmgr - Manager for the subsystem.                        */
/*        PRBIOT   * pprbiot - Address of prbiot found.                     */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        PRBIOT * pprbiot      - Update To point to BIOS structure.        */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC ConnectorToBIOS(PSZ pszConnector,PRMGR prmgr,PRBIOT *pprbiot)
{

   PRBIOT     prbiot;                  // Handle to filename
   RC         ulRC = DSP_NOERROR;      // Assume DSP_NOERROR
   BOOL       bfound = FALSE;          // Assume FALSE

   /* Verify that the Connector name is valid                               */

   if (pszConnector == NULL)
      return (DSP_INV_PARAMETER);

   MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspload::ConnectorToBIOS entry pszConnector %s prmgr %x\n",
	   pszConnector,(int)prmgr);

   /* Traverse the MGR BIOT Table looking for the connector name.           */

   if (prmgr->MGR_prBiotTail != NULL) {
      prbiot = prmgr->MGR_prBiotTail;
      do {
         if (mwstrcmp(pszConnector, prbiot->BT_pszConnector) == 0) {
            bfound = TRUE;
            *pprbiot = prbiot;
         }
         else {
            prbiot = prbiot->BT_prBiotNext;
         }
      }  while ((prbiot != prmgr->MGR_prBiotTail) && (bfound == FALSE));
   }
   else
      ulRC = DSP_HWDEVICE_NOT_AVAILABLE;

   /* Return the pointer to the BIOT entry or an Insufficient resource error*/
   MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspload::ConnectorToBIOS exit ulRC %lx pprbiot %x\n",ulRC,(int)*pprbiot);

   if (bfound == FALSE)
      ulRC = DSP_NAME_NOT_FOUND;
   return (ulRC);
}

/**************************START OF SPECIFICATIONS **************************/
/*                                                                          */
/* SUBROUTINE NAME: GetBIOSTask                                             */
/*                                                                          */
/* FUNCTION: This routine searches for the Specified BIOS task in ISPOS or  */
/*           load it from the BIOS file.                                    */
/*                                                                          */
/* INPUT:                                                                   */
/*        PSZ         pszFileName - FileName.                               */
/*        PSZ         pszTaskName - TaskName.                               */
/*        PRMGR   prmgr - Manager for the subsystem.                        */
/*        PRTSK   * pprtask - Address of task found.                        */
/*                                                                          */
/* OUTPUT:                                                                  */
/*        PRBIOT * pprtask      - Update To point to Task structure.        */
/*        ULONG  ReturnCode   - 0 if no error                               */
/*                              !0 if error                                 */
/*                                                                          */
/* SIDE EFFECTS: (NONE)                                                     */
/*                                                                          */
/* MODIFICATION HISTORY:                                                    */
/*     DATE      NAME  CHANGE DESCRIPTION                                   */
/*                                                                          */
/************************** END OF SPECIFICATIONS ***************************/

RC GetBIOSTask(PSZ pszFileName,PSZ pszTaskName,PRMOD prmod,PRDSP prdsp,
               PRTSK *pprtask)
{

   PRTSK      prtask;                  // Handle to task
   RC         ulRC;                    // Error return code



   MW_SYSLOG_5(TRACE_MANAGER_CORE,"dspload::GetBIOSTask entry pszFileName %s pszTaskName %s prmod %x prdsp %x\n",
	   pszFileName,pszTaskName,(int)prmod,(int)prdsp);
   /* Verify that the FileName and TaskName is valid                        */

   if ((pszFileName == NULL) || (pszTaskName == NULL))
      return (DSP_INV_PARAMETER);

   ulRC = NameToTaskHandle(prmod, pszTaskName, &prtask);

   /* If task is not loaded, load it in ACTIVE.                             */

   if (ulRC == DSP_NAME_NOT_FOUND) {
     ulRC = LoadTask(prmod, prdsp, pszFileName, pszTaskName, pszTaskName,
                     DSP_ACTIVATE_STANDBY, NULL, &prtask);

      if (ulRC == DSP_NOERROR) {
         ulRC = ChangeTaskState(prtask, DSP_ACTIVATE_ACTIVE);
         *pprtask = prtask;
      }
   }
   else
      *pprtask = prtask;

   MW_SYSLOG_3(TRACE_MANAGER_CORE,"dspload::GetBIOSTask exit ulRC %lx pprtask %x\n",ulRC,(int)*pprtask);
   return (ulRC);
}

