/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */
// $Id: QTAtom_stsc.cpp,v 1.7 1999/04/15 22:44:54 malyn Exp $
//
// QTAtom_stsc:
//   The 'stsc' QTAtom class.


// -------------------------------------
// Includes
//
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "QTFile.h"

#include "QTAtom.h"
#include "QTAtom_stsc.h"



// -------------------------------------
// Constants
//
const int		stscPos_VersionFlags		=  0;
const int		stscPos_NumEntries			=  4;
const int		stscPos_SampleTable			=  8;



// -------------------------------------
// Macros
//
#define DEBUG_PRINT(s) if(fDebug) printf s
#define DEEP_DEBUG_PRINT(s) if(fDeepDebug) printf s



// -------------------------------------
// Class state cookie
//
QTAtom_stsc_SampleTableControlBlock::QTAtom_stsc_SampleTableControlBlock(void)
{
	Reset();
}

QTAtom_stsc_SampleTableControlBlock::~QTAtom_stsc_SampleTableControlBlock(void)
{
}

void QTAtom_stsc_SampleTableControlBlock::Reset(void)
{
	fCurEntry = 0;
	fCurSample = 1;
	fLastFirstChunk = 1;
	fLastSamplesPerChunk = 1;
	fLastSampleDescription = 0;
}



// -------------------------------------
// Constructors and destructors
//
QTAtom_stsc::QTAtom_stsc(QTFile * File, QTFile::AtomTOCEntry * TOCEntry, bool Debug, bool DeepDebug)
	: QTAtom(File, TOCEntry, Debug, DeepDebug),
	  fNumEntries(0), fSampleToChunkTable(NULL)
{
}

QTAtom_stsc::~QTAtom_stsc(void)
{
	//
	// Free our variables.
	if( fSampleToChunkTable != NULL )
		delete[] fSampleToChunkTable;
}



// -------------------------------------
// Initialization functions
//
bool QTAtom_stsc::Initialize(void)
{
	// Temporary vars
	UInt32		tempInt32;


	//
	// Parse this atom's fields.
	ReadInt32(stscPos_VersionFlags, &tempInt32);
	fVersion = (UInt8)((tempInt32 >> 24) & 0x000000ff);
	fFlags = tempInt32 & 0x00ffffff;

	ReadInt32(stscPos_NumEntries, &fNumEntries);

	//
	// Validate the size of the sample table.
	if( (unsigned long)(fNumEntries * 12) != (fTOCEntry.AtomDataLength - 8) )
		return false;

	//
	// Read in the sample-to-chunk table.
	fSampleToChunkTable = NEW_ARRAY('_tsc', char, fNumEntries * 12);
	if( fSampleToChunkTable == NULL )
		return false;
	
	ReadBytes(stscPos_SampleTable, fSampleToChunkTable, fNumEntries * 12);

	//
	// This atom has been successfully read in.
	return true;
}



// -------------------------------------
// Accessors
//
bool QTAtom_stsc::SampleNumberToChunkNumber(UInt32 SampleNumber, UInt32 *ChunkNumber, UInt32 *SampleDescriptionIndex, UInt32 *SampleOffsetInChunk, QTAtom_stsc_SampleTableControlBlock * STCB)
{
	// Temporary state var
	QTAtom_stsc_SampleTableControlBlock	tempSTCB;
	
	// General vars
	UInt32		NewCurSample;
	UInt32		FirstChunk = 0, SamplesPerChunk = 0, SampleDescription = 0;


	//
	// Use a temporary STCB if we weren't given one.  We cannot use a default
	// STCB as we would have no way to know when we are seeking around in the
	// movie.
	if( STCB == NULL )
		STCB = &tempSTCB;

	//
	// Assume that this sample came out of the last chunk.
	if( ChunkNumber != NULL )
		*ChunkNumber = STCB->fLastFirstChunk + ((SampleNumber - STCB->fCurSample) / STCB->fLastSamplesPerChunk) ;
	if( SampleDescriptionIndex != NULL )
		*SampleDescriptionIndex = STCB->fLastSampleDescription;
	if( SampleOffsetInChunk != NULL )
		*SampleOffsetInChunk = SampleNumber - (STCB->fCurSample + ((*ChunkNumber - STCB->fLastFirstChunk) * STCB->fLastSamplesPerChunk));
	
	//
	// Linearly search through the sample table until we find the chunk
	// which contains the given sample.
	for( ; STCB->fCurEntry < fNumEntries; STCB->fCurEntry++ ) {
		//
		// Copy this entry's fields.
		memcpy(&FirstChunk, fSampleToChunkTable + (STCB->fCurEntry * 12) + 0, 4);
		FirstChunk = ntohl(FirstChunk);
		memcpy(&SamplesPerChunk, fSampleToChunkTable + (STCB->fCurEntry * 12) + 4, 4);
		SamplesPerChunk = ntohl(SamplesPerChunk);
		memcpy(&SampleDescription, fSampleToChunkTable + (STCB->fCurEntry * 12) + 8, 4);
		SampleDescription = ntohl(SampleDescription);
		
		//
		// Check to see if the sample was actually in the last chunk and
		// return if it was.
		NewCurSample = STCB->fCurSample + (FirstChunk - STCB->fLastFirstChunk) * STCB->fLastSamplesPerChunk;
		if( SampleNumber < NewCurSample )
			return true;
		STCB->fCurSample = NewCurSample;
		
		//
		// Assume that the sample will be in this chunk.
		if( ChunkNumber != NULL )
			*ChunkNumber = FirstChunk + ((SampleNumber - STCB->fCurSample) / SamplesPerChunk) ;
		if( SampleDescriptionIndex != NULL )
			*SampleDescriptionIndex = SampleDescription;
		if( SampleOffsetInChunk != NULL )
			*SampleOffsetInChunk = SampleNumber - (STCB->fCurSample + ((*ChunkNumber - FirstChunk) * SamplesPerChunk));
		
		//
		// We have yet to find the sample; update our CurSample counter
		// and move on.
		STCB->fLastFirstChunk = FirstChunk;
		STCB->fLastSampleDescription = SampleDescription;
		STCB->fLastSamplesPerChunk = SamplesPerChunk;
	}
	
	//
	// Falling out of the loop means that the sample is in the last chunk of
	// the table.
	return true;
}



// -------------------------------------
// Debugging functions
//
void QTAtom_stsc::DumpAtom(void)
{
	DEBUG_PRINT(("QTAtom_stsc::DumpAtom - Dumping atom.\n"));
	DEBUG_PRINT(("QTAtom_stsc::DumpAtom - ..Number of STC entries: %ld\n", fNumEntries));
}

void QTAtom_stsc::DumpTable(void)
{
	// General vars
	UInt32		FirstChunk = 0, SamplesPerChunk = 0, SampleDescription = 0;


	//
	// Print out a header.
	printf("-- Sample-to-Chunk table -------------------------------------------------------\n");
	printf("\n");
	printf("  Sample Num   FirstChunk  Samples/Chunk  Sample Description\n");
	printf("  ----------   ----------  -------------  ------------------\n");
	
	//
	// Print the table.
	for( UInt32 CurEntry = 0; CurEntry < fNumEntries; CurEntry++ ) {
		//
		// Copy this entry's fields.
		memcpy(&FirstChunk, fSampleToChunkTable + (CurEntry * 12) + 0, 4);
		FirstChunk = ntohl(FirstChunk);
		memcpy(&SamplesPerChunk, fSampleToChunkTable + (CurEntry * 12) + 4, 4);
		SamplesPerChunk = ntohl(SamplesPerChunk);
		memcpy(&SampleDescription, fSampleToChunkTable + (CurEntry * 12) + 8, 4);
		SampleDescription = ntohl(SampleDescription);
		
		//
		// Print out a listing.
		printf("  %10lu : %10lu   %10lu        %10lu\n",
			   CurEntry+1, FirstChunk, SamplesPerChunk, SampleDescription);
	}
}
