/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */
/*
	File:		OSQueue.h

	Contains:	implements OSQueue class
					
	$Log: OSQueue.h,v $
	Revision 1.2  1999/02/19 23:06:14  ds
	Created
		
	
*/

#ifndef _OSQUEUE_H_
#define _OSQUEUE_H_

#include "MyAssert.h"
#include "OSHeaders.h"
#include "OSMutex.h"
#include "OSCond.h"
#include "OSThread.h"

#define OSQUEUETESTING 0

class OSQueue;

class OSQueueElem {
	public:
		OSQueueElem(void* enclosingObject = NULL) : fNext(NULL), fPrev(NULL), fQueue(NULL),
													fEnclosingObject(enclosingObject) {}
		virtual ~OSQueueElem() { Assert(fQueue == NULL); }

		bool IsMember(const OSQueue& queue) { return (&queue == fQueue); }
		bool IsMemberOfAnyQueue() 	{ return fQueue != NULL; }
		void* GetEnclosingObject() 	{ return fEnclosingObject; }
		void SetEnclosingObject(void* obj) { fEnclosingObject = obj; }

		OSQueueElem* Next() { return fNext; }
		OSQueueElem* Prev() { return fPrev; }
		OSQueue* InQueue()	{ return fQueue; }
		inline void	Remove();

	private:

		OSQueueElem*	fNext;
		OSQueueElem*	fPrev;
		OSQueue *		fQueue;
		void* 			fEnclosingObject;

		friend class 	OSQueue;
};

class OSQueue {
	public:
		OSQueue();
		~OSQueue() {}

		void 			EnQueue(OSQueueElem* object);
		OSQueueElem* 	DeQueue();

		OSQueueElem* 	GetHead() { if (fLength > 0) return fSentinel.fPrev; return NULL; }
		OSQueueElem* 	GetTail() { if (fLength > 0) return fSentinel.fNext; return NULL; }
		UInt32 			GetLength() { return fLength; }
		
		void 			Remove(OSQueueElem* object);

#if OSQUEUETESTING
		static bool		Test();
#endif

	protected:

		OSQueueElem		fSentinel;
		UInt32 			fLength;
};

class OSQueueIter
{
	public:
		OSQueueIter(OSQueue* inQueue) : fQueueP(inQueue), fCurrentElemP(inQueue->GetHead()) {}
		~OSQueueIter() {}
		
		void			Reset() { fCurrentElemP = fQueueP->GetHead(); }
		
		OSQueueElem*	GetCurrent() { return fCurrentElemP; }
		void			Next();
		
		bool			IsDone() { return fCurrentElemP == NULL; }
		
	private:
	
		OSQueue*		fQueueP;
		OSQueueElem*	fCurrentElemP;
};

class OSQueue_Blocking
{
	public:
		OSQueue_Blocking() : fMutex('blkm') {}
		~OSQueue_Blocking() {}
		
		OSQueueElem* 	DeQueueBlocking(OSThread* inCurThread, SInt32 inTimeoutInMilSecs);
		OSQueueElem*	DeQueue();//will not block
		void			EnQueue(OSQueueElem* obj);
		
		OSCond*			GetCond() 	{ return &fCond; }
		OSQueue*		GetQueue() 	{ return &fQueue; }
		
	private:

		OSCond				fCond;
		OSMutex				fMutex;
		OSQueue			 	fQueue;
};


void	OSQueueElem::Remove()
{
	if (fQueue != NULL)
		fQueue->Remove(this);
}
#endif //_OSQUEUE_H_