#!/bin/sh

# apache2-block-ip.sh: manually block/unblock IPs for incomming HTTP(S) in firewall
# (c) 2015 Jonas Meurer / INTERNET AG <jmeurer@inet.de>
#
# usage:
#
# # monitor situation with apachetop:
# (1) monitor all access/custom logs (keep open in separate terminal):
#     # apachetop $(apache2-logs.sh custom | sed -e 's/ \// -f \//g' -e 's/^/-f /g')
#
# # block IPs with more than 5 bad requests to Apache2:
# 
# (2) log IPs with bad requests against Apache2 to a file (keep open in separate terminal):
#     # tail -f $(apache2-logs.sh error) | sed -ne 's/^.*\[client \([0-9\.]\+\)\].*$/\1/gp' >>/tmp/apache2-bad-ips.log
# (3) run this script to block all IPs that have more than 5 (bad) requests:
#     # sort /tmp/apache2-bad-ips.log | uniq -c | sort -rn | grep -v "^\s*[1-4]\s\+" | awk '{print $2}' | xargs -n1 -I{} apache2-block-ip.sh {} block
#
#  or
#
# # block the 20 IPs with most requests (any) to Apache2:
#
# (2) log IPs with any requests against Apache2 to a file (keep open in separate terminal):
#     # tail -f $(apache2-logs.sh custom) | sed -ne 's/^\([0-9\.]\+\)\s\+.*$/\1/gp' >>/tmp/apache2-bad-ips.log
# (3) run this script to block the 20 IPs with most requests:
#     # sort /tmp/apache2-bad-ips.log | uniq -c | sort -rn | head -n 20 | awk '{print $2}' | xargs -n1 -I{} apache2-block-ip.sh {} block

if [ -z "$1" ]; then
	echo "Error: missing IP as argument" >&2
	echo "Usage: $0 <IP> {block|start|unblock|stop|unblockall|stopall|check|status|whitelist}" >&2
	exit 1
fi

IP="$1"
IPLOG="/tmp/apache2-block-ip.log"

if [ -f "/etc/fail2ban/jail.conf" ]; then
	ignorenets="$(sed -n -e 's/^ignoreip\s*=\s*//gp' /etc/fail2ban/jail.conf)"
else
	ignorenets=""
fi

if [ -n "$2" ]; then
	case "$2" in
	  block|start)
	  ;;
	  unblock|stop)
		if grep -q "$IP" "$IPLOG"; then
			iptables -D fail2ban-apache-notfound -s $IP -j DROP
			sed -i "/$IP/d" $IPLOG
			echo "Info: IP $IP unblocked"
			exit 0
		else
			echo "Warning: IP $IP not blocked"
			exit 1
		fi
	  ;;
	  unblockall|stopall)
		while read aggip; do
			$0 $aggip unblock
		done <$IPLOG
	  ;;
	  check|status)
		if grep -q "$IP" "$IPLOG"; then
			echo "Status: IP $IP blocked"
		else
			echo "Status: IP $IP not blocked"
		fi
		exit 0
	  ;;
	  whitelist)
		while read bip; do
			if [ "$(perl -e 'use Net::Subnets; my @subnets = qw('"$ignorenets"'); my $sn = Net::Subnets->new; $sn->subnets(\@subnets); my $address="'$bip'"; if ($sn->check(\$address)) { print "yes\n"; }')" = "yes" ]; then
				echo "Warning: IP $bip blocked while whitelisted"
				$0 "$bip" unblock
			fi
		done <"$IPLOG"
		exit 0
	  ;;
	  *)
		echo "Error: unknown action: $2" >&2
		echo "Usage: $0 <IP> {block|start|unblock|stop|unblockall|stopall|check|status|whitelist}" >&2
		exit 1
	  ;;
	esac
fi

if ! grep -q "$IP" "$IPLOG"; then
	if [ "$(perl -e 'use Net::Subnets; my @subnets = qw('"$ignorenets"'); my $sn = Net::Subnets->new; $sn->subnets(\@subnets); my $address="'$IP'"; if ($sn->check(\$address)) { print "yes\n"; }')" = "yes" ]; then
		echo "Warning: IP $IP whitelisted"
		exit 0
	fi
	iptables -I fail2ban-apache-notfound 1 -s $IP -j DROP
	echo "$IP" >>$IPLOG
	echo "Info: IP $IP blocked"
	exit 0
else
	#echo "Warning: IP $IP already blocked"
	exit 1
fi
