#!/usr/bin/perl -w
#
# 2006 (C) Andrei Warkentin, Data Armor. 
# Read the GNU Copyright stuff for all the legalese.
# 
# Check 3ware array health. Uses the 3ware tw_cli utility.
# This is based of check_ntp.pl plugin, by Ian Cass.

use strict;
use Getopt::Long;

use Digest::MD5 qw(md5_hex);

# Note that we don't use the critical, timeout or warning flags for anything. Or the hostname.
use vars qw($state $state_message $PROGNAME $opt_controller $opt_version $opt_verbose $opt_help $opt_hostname $opt_warning $opt_critical $opt_timeout);
use lib qw(/usr/lib/nagios/plugins /usr/lib64/nagios/plugins /usr/local/nagios/libexec);
use utils qw($TIMEOUT %ERRORS &print_revision &support);

# Execute command for running the 3ware CLI.
my $tw_cli = "/usr/sbin/tw_cli";

# Statusfile with list of md5sums to be ignored
my $statusfile = "/tmp/nagios_3ware_errors.lst";

# Name of this plugin.
$PROGNAME = "check_3ware";

sub print_help();
sub print_usage();

# Clear the environment.
$ENV{'PATH'} = '';
$ENV{'BASH_ENV'} = '';
$ENV{'ENV'} = '';

Getopt::Long::Configure('bundling');
GetOptions
    (
     "V" => \$opt_version, "version" => \$opt_version,
     "h" => \$opt_help, "help" => \$opt_help,
     "v" => \$opt_verbose, "verbose" => \$opt_verbose,
     "w=s" => \$opt_warning, "warning=s" => \$opt_warning,
     "c=s" => \$opt_critical, "critical=s" => \$opt_critical,
     "t=s" => \$opt_timeout, "timeout=s" => \$opt_timeout,
     "H=s" => \$opt_hostname, "hostname=s" => \$opt_hostname,
     "d=s" => \$opt_controller, "controller=s" => \$opt_controller);

if($opt_version) {
    print_revision($PROGNAME, 'Version: 1.0');
    exit $ERRORS{'OK'};
}
     
if($opt_help) {
    print_help();
    exit $ERRORS{'OK'};
}

if(! $opt_controller) {
    print "ERROR Missing controller name - did you pass anything to -d?\n";
    exit $ERRORS{"UNKNOWN"};
}

# Check for tw_cli presence.
if(! -x $tw_cli) {
    print "ERROR No executable tw_cli at $tw_cli.\n";
    exit $ERRORS{"UNKNOWN"};
}

# First parse just general info.
if(!open(TWCLI, $tw_cli . " info $opt_controller 2>&1 |")) {
    print "ERROR Could not open $tw_cli: $!\n";
    exit $ERRORS{"UNKNOWN"};
}
$state = 'UNKNOWN';
$state_message = 'No understandable output from  tw_cli';
my $out;

# There is some reasoning behind the ordering of the checks.
while(<TWCLI>)
{
    $out .= " $_";
    if(/OK/)
    {
        $state = 'OK';
        $state_message = "Arrays OK on Controller $opt_controller";       
    }
    if(/VERIFYING/)
    {
        $state = 'OK';
	$state_message = "Arrays Verifying on Controller $opt_controller";
    }
    if(/INITIALIZING/)
    {
        $state = 'WARNING';
        $state_message = "Arrays Initializing on Controller $opt_controller";
    }
    if(/REBUILDING/)
    {
        $state = 'WARNING';
        $state_message = "Arrays Rebuilding on Controller $opt_controller";
    }
    if(/MIGRATING/)
    {
        $state = 'WARNING';
        $state_message = "Arrays Migrating on Controller $opt_controller"; 
    } 
    if(/DEGRADED/) {
        $state = 'CRITICAL';
        $state_message = "Arrays Degraded on Controller $opt_controller";
    }     
    if(/INOPERABLE/)
    {
        $state = 'CRITICAL'; 
        $state_message = "Arrays Inoperable on Controller $opt_controller";
    }
}
close(TWCLI) || 
    die $! ? "$out - Error close $tw_cli pipe: $!"
    : "$out - Exit status: $? from $tw_cli\n";

# Parse the alarm info.
if(!open(TWCLI, $tw_cli . " /$opt_controller show alarms 2>&1 |")) {
    print "Could not open $tw_cli: $!\n";
    exit $ERRORS{"UNKNOWN"};
}

my @errors_md5;
if(open(MD5LST, "$statusfile")) {
    while (<MD5LST>)
    {
        if(/^[^#]/)
        {
            chomp($_);
            push(@errors_md5, $_);
        }
    }
    close(MD5LST);
}

$out .= " Events on Controller $opt_controller (only WARNINGS and ERRORS):\n\n";
$out .= "   In order to ignore (acknowledge) ERRORS, add md5sum to $statusfile:\n";
$out .= "   # echo [MD5SUM_FROM_LIST_BELOW] >> $statusfile\n";

my $md5;
my $ignored = 'yes';
my @errors_md5_found;
while(<TWCLI>)
{
    if(/INFO/)
    {
        #$out .= " $_";
    }
    elsif(/WARNING/)
    {
        $out .= " $_";
    }
    elsif(/ERROR/)
    {
        chomp($_);
        $md5 = md5_hex($_);
        if ( grep { $_ eq $md5 } @errors_md5 )
        {
            push(@errors_md5_found, $md5);
            $out .= " $_ (md5: $md5 [ignored])\n";
        }
        else
        {
            $out .= " $_ (md5: $md5)\n";
            $state = 'CRITICAL'; 
            $ignored = 'no';
        }
        if ($ignored eq 'no')
        {
            $state_message = "Alarms on Controller $opt_controller";
        }
        else
        {
            $state_message = "Alarms on Controller $opt_controller (but ignored)";
        }
    }
    else
    {
        $out .= " $_";
    }
}
close(TWCLI) || 
    die $! ? "$out - Error close $tw_cli pipe: $!"
    : "$out - Exit status: $? from $tw_cli\n";

print "$state_message. | STATUS=$ERRORS{$state}\n";
if($opt_verbose)
{
    print $out;
    my $md5info;
    foreach my $md5 (@errors_md5)
    {
        if ( ! grep { $_ eq $md5 } @errors_md5_found )
        {
            $md5info .= "        $md5\n";
        }
    }
    if ($md5info)
    {
        print "INFO: unknown md5 sums in $statusfile:\n";
        print "$md5info";
        print "\n";
    }
}
exit $ERRORS{$state};

sub print_usage()
{
    print "Usage: $PROGNAME [-d array controller] [-v verbose]\n";
}
 
sub print_help() 
{
    print_revision($PROGNAME, "Version: 1.1");
    print "Copyright (C) 2006 Andrei Warkentin\n";
    print "\n";
    print_usage();
    print "Checks the status of the 3ware controller (c0, c1, etc) passed with -d.\n\n";
    support();
}
