#! /usr/bin/perl -w

# Check for newer upstream releases. Requires configuration below.
# Copyright 2012-2014 Jonas Meurer <jmeurer@inet.de>
#
# Permission is hereby granted, free of charge, to any person obtaining
# a copy of this software and associated documentation files (the
# "Software"), to deal in the Software without restriction, including
# without limitation the rights to use, copy, modify, merge, publish,
# distribute, sublicense, and/or sell copies of the Software, and to
# permit persons to whom the Software is furnished to do so, subject to
# the following conditions:
#
# The above copyright notice and this permission notice shall be
# included in all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
# LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
# OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
# WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.


###############################################################################
###   CONFIGURATION                                                         ###
###############################################################################

###############################################################################
### Status if local version is more recent than latest upstream release     ###
### Possible values: 0 = OK | 1 = WARNING                                   ###
$warn_l_gt_r	= 1;
###############################################################################

###############################################################################
###   Examples:                                                             ### 
###############################################################################
#$package	= 'Roundcube'                                                 #
#$changelog	= '/usr/src/roundcubemail-0.8.4/CHANGELOG';                   #
#$url		= 'http://sourceforge.net/projects/roundcubemail/files/roundcubemail/';
#$regex		= 'roundcubemail/files/roundcubemail/([^v]__VER__)/';         #
###############################################################################
#$package	= 'Linux Kernel'                                              #
#$changelog	= '/usr/src/linux/ChangeLog-3.6.1';                           #
#$url		= 'ftp://ftp.kernel.org/pub/linux/kernel/v3.x/';              #
#$regex		= 'linux-(__VER__)\.tar';                                     #
###############################################################################
$package	= 'Roundcube';
$changelog	= '/usr/local/share/roundcube/roundcubemail-current/CHANGELOG';
$url		= 'http://sourceforge.net/projects/roundcubemail/files/roundcubemail/';
$regex		= 'roundcubemail/files/roundcubemail/([^v]__VER__)/';
###############################################################################

###############################################################################
###   END CONFIGURATION                                                     ###
###############################################################################

glob $urlfile = "/tmp/nagios-check-release-$package.page";
glob $maxage = '1';

my %ERRORS=( OK => 0, WARNING => 1, CRITICAL => 2, UNKNOWN => 3 );
my %ERRORCODES=( 0 => 'OK', 1 => 'WARNING', 2 => 'CRITICAL', 3 => 'UNKNOWN' );

glob $package; glob $changelog; glob $url; glob $regex;

if (!$package || !$changelog || !$url || !$regex || !$urlfile || !$maxage) {
    print "CRITICAL: Configuration error: Unset variable\n";
    exit $ERRORS{'CRITICAL'};
}


use File::Basename;
use Dpkg::Version;
require LWP::UserAgent;
require HTTP::Request;

my $l_vers = get_local_version();
my $u_vers = get_upstream_version();

if (version_compare($l_vers, $u_vers) == "-1") {
    print "WARNING $package: New release available: $u_vers, installed version: $l_vers (URL: $url) | STATUS=1\n";
    exit $ERRORS{'WARNING'};
} elsif (version_compare($l_vers, $u_vers) == "0") {
    print "OK $package: Newest release installed $u_vers | STATUS=0\n";
    exit $ERRORS{'OK'};
} elsif (version_compare($l_vers, $u_vers) == "1") {
    if ($warn_l_gt_r) {
         print "WARNING $package: Installed release newer ($l_vers) than available upstream ($u_vers) | STATUS=1\n";
         exit $ERRORS{'WARNING'};
    } else {
         print "OK $package: Installed release newer ($l_vers) than available upstream ($u_vers) | STATUS=0\n";
         exit $ERRORS{'OK'};
    }
}

print "installed version: $l_vers\n";
print "available version: $u_vers\n";

sub get_local_version {
    my @diag = ();
    my @ch_vers = ();

    open(my $in, '<', $changelog) or die "Could not open ($changelog) File";
    while (<$in>) {
        chomp;
        if (/^(\d|v\d)/) {
        # Common
            my ($cvers, $date) = split(/\s+/, $_, 2);
            push (@ch_vers, "$cvers");
        } elsif (/^\s+version: ([\d.]+)$/) {
        # YAML
            push (@ch_vers, "$1");
        } elsif (/^\* (v?[\d._]+)$/) {
        # Apocal
            push (@ch_vers, "$1");
        #} elsif (/^Version (v?[\d._]+)($|[:,[:space:]])/i) {
        } elsif (/^Version (v?[\S]+)($|[:,[:space:]])/i) {
        # Plain "Version N"
            push (@ch_vers, "$1");
        #} elsif (/^Release (v?[\d._]+)($|[:,[:space:]])/i) {
        } elsif (/^Release (v?[\S]+)($|[:,[:space:]])/i) {
        # Plain "Release N"
            push (@ch_vers, "$1");
        } elsif (/^\s*Linux (v?[\d._]+)($|[:,[:space:]])/i) {
        # Linux Kernel "Linux N"
            push (@ch_vers, "$1");
        }
    }
    close($in) or die "Could not close ($changelog) file";

    if (! @ch_vers) {
        print "CRITICAL: Unable to get version from changelog file ($changelog) | STATUS=2\n";
        exit $ERRORS{'CRITICAL'};
    }

    # get most recent version from changelog
    my $vers_recent = '0';
    for my $vers (@ch_vers) {
        if (version_compare($vers, $vers_recent) == 1) {
            $vers_recent = $vers;
        }
    }

    if ($vers_recent eq '0') {
        print "CRITICAL: Unable to get version from changelog file ($changelog) | STATUS=2\n";
        exit $ERRORS{'CRITICAL'};
    }

    return $vers_recent;
}

sub get_upstream_version {
    my @diag = ();
    my @up_vers = ();

    my $isfile = -e $urlfile;
    my $fileage = -M $urlfile;
    #print "urlfile age: $fileage\n";
    if (!$isfile || $fileage gt "$maxage") {
        #print "urlfile to old, fetching new\n";
        open(my $out, '>', $urlfile) or die "Could not open ($urlfile) File";
        print $out fetch_url();
        close($out) or die "Could not close ($urlfile) file";
    }

    # version parsing regex templates and suchlike
    my %sub_vers = ();
    my $no = 0;
    for (qw(final rc pre+prerelease+preview+test omega epsilon delta gamma beta alpha)) {
        @sub_vers{split /\+/} = (--$no) x (@{[/\+/g]}+1)
    }
    my $re_ver = "(?i:[0-9._-]|(?<![a-z])(?:[a-z]|@{[join '|', keys %sub_vers]})(?![a-z]))+";
    $regex =~ s/__VER__/$re_ver/g;
    #print "regex: $regex\n";

    open(my $in, '<', $urlfile) or die "Could not open ($urlfile) File";
    while (<$in>) {
        chomp;
        if (/$regex/) {
            push (@url_vers, "$1");
        }
    }
    close($in) or die "Could not close ($urlfile) file";

    if (! @url_vers) {
        print "CRITICAL: Unable to get versions from file ($urlfile) | STATUS=2\n";
        exit $ERRORS{'CRITICAL'};
    }

    # get most recent version from $url_data
    my $vers_recent = '0';
    for my $vers (@url_vers) {
        if (version_compare($vers, $vers_recent) == 1) {
            $vers_recent = $vers;
        }
    }

    return $vers_recent;
}

sub fetch_url {
    my $xfersum = 0;  # bytes transferred
    my $ua = new LWP::UserAgent;
    $ua->agent("nagios-check_release");
    my $req = HTTP::Request->new(GET => $url);
    local $_ = $ua->request($req);
    $xfersum += length($_->headers) + length $_->content;
    $_->content
}

