#!/usr/bin/perl
# nagios: -epn

##########################################################################
#
# check_cisco_ace_slb - script for checking service load balancer
# on Cisco ACE 4710 (clusters) using SNMP
#
# URL: https://git.inet.de/?p=monitoring-plugins-inet.git;a=tree;f=server
#
##########################################################################
#
# required perl packages: libsnmp-perl libnet-snmp-perl
# required SNMP MIB files: CISCO-ENHANCED-SLB-MIB and dependencies
#
##########################################################################
#
#  (c) 2015  JonasMeurer <jmeurer@inet.de>
#            INTERNET AG Global Networks
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty
# of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# you should have received a copy of the GNU General Public License
# along with this program (or with Netsaint);  if not, write to the
# Free Software Foundation, Inc., 59 Temple Place - Suite 330,
# Boston, MA 02111-1307, USA
#
##########################################################################
#
# Version: 0.3
# Changelog: 
#	2015-06-08: Basic checks for RServers in SerferFarm implemented.
#                   First usable version.
#	2015-05-27: Import of RServers, SFarms and VServers working.
# 	2015-05-13: Initial version created. Based on check_ace_failover
#                   by Christian Hailer / Interhyp AG.
#
##########################################################################

use strict;
use warnings;

use lib ('/usr/lib/nagios/plugins', '/usr/lib/interfacetable-v3t');
use Net::SNMP qw(oid_base_match);
use SNMP;
use Data::Dumper;
$Data::Dumper::Sortkeys = 1;
$Data::Dumper::Terse = 1;
use Getopt::Long;
use utils qw (%ERRORS $TIMEOUT);
use SnmpUtils;

my $hostname;
my $community;
my $debug;
my $timeout;
my $retries;
my $help;
# TODO: remove cachedir usage
my $cachedir = "/tmp/.aceCache";
my $refhSNMPResult;
my $refhSNMPResultRSs;
my $refhSNMPResultSFs;
my $refhSNMPResultSFRSs;
my $refhSNMPResultVSs;
my @RServers;
my @SFarms;
my @VServers;
my $pluginstate = "OK";
my $pout = "";
my $perfdata;

Getopt::Long::Configure('bundling');
GetOptions (
	"help" 		=> \$help,
	"hostname=s" 	=> \$hostname,
	"community=s" 	=> \$community,
	"debug" 	=> \$debug,
	"timeout=i"	=> \$timeout,
	"retries=i"	=> \$retries,
	"h" 		=> \$help,
	"H=s"	 	=> \$hostname,
	"C=s"	 	=> \$community,
	"d" 		=> \$debug,
	"t=i"		=> \$timeout,
	"r=i"		=> \$retries
);

unless (defined ($debug)) {
	$debug = 0;
}
unless (defined ($timeout)) {
	$timeout = $TIMEOUT;
}
unless (defined ($retries)) {
	$retries = 1;
}

# Status returned by the SNMP query

my %rserver_adm_status = (
	'1' => 'inService',
	'2' => 'outOfService',
	'3' => 'inServiceStandby',
);

my %rserver_oper_status = (
	'1' => 'outOfService',
	'2' => 'inService',
	'3' => 'failed',
	'4' => 'readyToTest',
	'5' => 'testing',
	'6' => 'maxConnsThrottle',
	'7' => 'maxClientsThrottle',
	'8' => 'dfpThrottle',
	'9' => 'probeFailed',
	'10' => 'probeTesting',
	'11' => 'operWait',
	'12' => 'testWait',
	'13' => 'inbandProbeFailed',
	'14' => 'returnCodeFailed',
	'15' => 'arpFailed',
	'16' => 'standby',
	'17' => 'inactive',
	'18' => 'maxLoad',
);

my %vserver_state = (
	'1' => 'outOfService',
	'2' => 'inService',
	'3' => 'standby',
	'4' => 'inOperReal',
	'5' => 'stbInOperReal',
	'6' => 'testReal',
	'7' => 'outOfMemory',
);

###########################################
# sub: help
# Prints help information

sub help () {
	print <<USE;
Usage: $0 [options]
where options is
-h, --help		Print this text
-H, --hostname		Hostname (required)
-C, --community		SNMP Community (required)
-d, --debug		Show debug information
-t, --timeout		Timeout value in seconds (defaults to $TIMEOUT)
-r, --retries		Number of retries (defaults to 1)

Note that every retry has its own timeout value, for example,
if timeout is 15 and retries is 1, maximum timeout would be 30s.

USE
	exit ($ERRORS{OK});
};

# Check command line arguments

&help () if ($help);
&help () unless ($hostname && $community);

# Set defaults for SNMP connection
my %hOptionDefaults = (
  'checkempty' => 1,
  'community' => $community,
  'domain' => 'udp',
  'host' => $hostname,
  'port' => 161,
  'retries' => $retries,
  'timeout' => $timeout,
  'version' => '2c',
  'cachetimer' => 0,
  'cachedir' => "$cachedir",
  'outputhashkeyidx' => 0,
);
my %hOptions = ();

# general OIDs
my %oid_sysUpTime         = ( name => "sysUpTime",      mib => "RFC1213-MIB",   oid => "1.3.6.1.2.1.1.3.0" );
# RServer OIDs
my %oid_RServerDesc       = ( name => "RServerDesc",       mib => "CISCO-ENHANCED-SLB-MIB", oid => "1.3.6.1.4.1.9.9.470.1.1.1.1.5" );
my %oid_RServerAdmStatus  = ( name => "RServerAdmStatus",  mib => "CISCO-ENHANCED-SLB-MIB", oid => "1.3.6.1.4.1.9.9.470.1.1.1.1.12" );
my %oid_RServerOperStatus = ( name => "RServerOperStatus", mib => "CISCO-ENHANCED-SLB-MIB", oid => "1.3.6.1.4.1.9.9.470.1.1.1.1.13" );
my %oid_RServerStateDesc  = ( name => "RServerStateDesc",  mib => "CISCO-ENHANCED-SLB-MIB", oid => "1.3.6.1.4.1.9.9.470.1.1.1.1.14" );
my %oid_RServerNumOfCons  = ( name => "RServerNumOfCons",  mib => "CISCO-ENHANCED-SLB-MIB", oid => "1.3.6.1.4.1.9.9.470.1.1.1.1.19" );
my %oid_RServerTotalCons  = ( name => "RServerTotalCons",  mib => "CISCO-ENHANCED-SLB-MIB", oid => "1.3.6.1.4.1.9.9.470.1.1.1.1.17" );
# ServerFarm OIDs
my %oid_SFarmRSvrAStatus  = ( name => "SFarmRSvrAStatus",  mib => "CISCO-ENHANCED-SLB-MIB", oid => "1.3.6.1.4.1.9.9.470.1.1.3.1.6" );
my %oid_SFarmRSvrOStatus  = ( name => "SFarmRSvrOStatus",  mib => "CISCO-ENHANCED-SLB-MIB", oid => "1.3.6.1.4.1.9.9.470.1.1.3.1.7" );
my %oid_SFarmRSvrStDesc   = ( name => "SFarmRSvrStDesc",   mib => "CISCO-ENHANCED-SLB-MIB", oid => "1.3.6.1.4.1.9.9.470.1.1.3.1.8" );
my %oid_SFarmRSvrTotCons  = ( name => "SFarmRSvrTotCons",  mib => "CISCO-ENHANCED-SLB-MIB", oid => "1.3.6.1.4.1.9.9.470.1.1.3.1.11" );
my %oid_SFarmRSvrNumCons  = ( name => "SFarmRSvrNumCons",  mib => "CISCO-ENHANCED-SLB-MIB", oid => "1.3.6.1.4.1.9.9.470.1.1.3.1.14" );
# VServers OIDs
my %oid_VServerClassMap   = ( name => "VserverClassMap",   mib => "CISCO-SLB-MIB", oid => "1.3.6.1.4.1.9.9.161.1.4.2.1.2" );
my %oid_VServerState      = ( name => "VServerState",      mib => "CISCO-SLB-MIB", oid => "1.3.6.1.4.1.9.9.161.1.4.2.1.4" );
my %oid_VServerStateDesc  = ( name => "VServerStateDesc",  mib => "CISCO-SLB-MIB", oid => "1.3.6.1.4.1.9.9.161.1.4.2.1.5" );
my %oid_VServerNumOfCons  = ( name => "VServerNumOfCons",  mib => "CISCO-SLB-MIB", oid => "1.3.6.1.4.1.9.9.161.1.4.2.1.6" );
my %oid_VServerTotalCons  = ( name => "VServerTotalCons",  mib => "CISCO-SLB-MIB", oid => "1.3.6.1.4.1.9.9.161.1.4.2.1.7" );

# Test SNMP connection by requesting uptime from host
eval {
  %hOptions = ( %hOptionDefaults, ( 'oids' => [ "$oid_sysUpTime{'oid'}" ] ));
  $refhSNMPResult = GetDataWithSnmp(\%hOptions);
};
ExitPlugin("UNKNOWN", add_oid_details($@->message(), [ \%oid_sysUpTime ])) if ($@);

# Fetch RServers, SFarms and VServers
Get_RServers(\%oid_RServerDesc);
Get_SFarms(\%oid_SFarmRSvrAStatus);
Get_VServers(\%oid_VServerClassMap);

# Debug: print RServers, SFarms and VServers
#print "RServers: ".Dumper(@RServers);
#print "SFarms: ".Dumper(@SFarms);
#print "VServers: ".Dumper(@VServers);

# check global RServers, not required and disabled for now
#my $rscnt=0;
#foreach my $rs_item (@RServers) {
#    $pout .= "RServers:" if ($rscnt == 0); $rscnt++;
#    $pout .= " (".$rs_item->{'Name'}.":";
#    Check_RServer($rs_item);
#    $pout .= ")";
#}
#$perfdata .= "RealServers=$rscnt ";

# check RServers in ServerFarms, this is important
my $sfcnt=0;
my $rs_active;
my $rs_active_full=0;
# error out when no ServerFarms are found
ExitPlugin("CRITICAL", "No ServerFarms found on Loadbalancer") unless (@SFarms);
foreach my $sfarm (@SFarms) {
    $rs_active = 0;
    $pout .= "ServerFarms: " if ($sfcnt == 0); $sfcnt++;
    $pout .= "{".$sfarm->{'Name'}.": ";
    my $sfrscnt=0;
    # critical if no RServers found for ServerFarm
    unless (@{$sfarm->{'RServers'}}) {
	$pluginstate="CRITICAL";
	$pout .= " Critical: No RServers found";
    }
    foreach my $sfrserver (@{$sfarm->{'RServers'}}) {
	$pout .= "RServers:" if ($sfrscnt == 0); $sfrscnt++;
	$pout .= " (".$sfrserver->{'Name'}.":";
	Check_RServer($sfrserver);
	$pout .= ")";
    }
    if ($rs_active == 0) {
	$pluginstate="CRITICAL";
	$pout .= " Critical: No active RServers found";
    }
    $pout .= "}";
    $rs_active_full += $rs_active;
}
$perfdata .= "ServerFarms=$sfcnt RealServersActive=$rs_active_full ";

#foreach (@VServers) {
#    $pout .= "VServer ".$_->{'ClassMap'}.": ";
#    if ($_->{'State'} != 2) {
#	$pout .= "Warning: Status: ".$vserver_state{$_->{'State'}}." ";
#    }
#    else {
#	$pout .= "Status: ".$vserver_state{$_->{'State'}}." ";
#	#$pout .= "Operational ";
#    }
#}

ExitPlugin($pluginstate, $pout);

# ------------------------------------------------------------------------------
# Get_RServers
# ------------------------------------------------------------------------------
# Description: gather details about the RServers in this context
# ------------------------------------------------------------------------------
sub Get_RServers {
    my $refhOIDRServers = shift;
    my %hOptions = ();

    # get all RServers
    eval {
        my %hOptions = ( %hOptionDefaults, (oids => ["$refhOIDRServers->{'oid'}"]));
        $refhSNMPResultRSs = GetTableDataWithSnmp (\%hOptions);
    };
    ExitPlugin("UNKNOWN", add_oid_details($@->message(), [ $refhOIDRServers ])) if ($@);

    while (my ($rsOID, $rsDesc) = each(%$refhSNMPResultRSs) ) {
	my @rsOIDEnds = split($refhOIDRServers->{'oid'}, $rsOID, 2);
	my $rsOIDEnd = $rsOIDEnds[1];
	%hOptions = ( %hOptionDefaults, (oids => [ "$oid_RServerAdmStatus{'oid'}$rsOIDEnd" ]));
	my $refhSNMPResultRSAdmStatus  = GetDataWithSnmp(\%hOptions);
	%hOptions = ( %hOptionDefaults, (oids => [ "$oid_RServerOperStatus{'oid'}$rsOIDEnd" ]));
	my $refhSNMPResultRSOperStatus = GetDataWithSnmp(\%hOptions);
	%hOptions = ( %hOptionDefaults, (oids => [ "$oid_RServerStateDesc{'oid'}$rsOIDEnd" ]));
	my $refhSNMPResultRSStateDesc  = GetDataWithSnmp(\%hOptions);
	%hOptions = ( %hOptionDefaults, (oids => [ "$oid_RServerNumOfCons{'oid'}$rsOIDEnd" ]));
	my $refhSNMPResultRSNCons      = GetDataWithSnmp(\%hOptions);
	%hOptions = ( %hOptionDefaults, (oids => [ "$oid_RServerTotalCons{'oid'}$rsOIDEnd" ]));
	my $refhSNMPResultRSTCons      = GetDataWithSnmp(\%hOptions);

	# translate RServerAdminStatus OID using MIB
	my $rsOIDTrans = TranslateOID("$oid_RServerAdmStatus{'oid'}$rsOIDEnd");
	# fetch ServerFarm name from translated OID name
	(my $rsName = $rsOIDTrans) =~ s/^CISCO-ENHANCED-SLB-MIB::cesRserverAdminStatus\.1\."([^"]+)".*$/$1/;
        push(@RServers, {
		'Name'       => $rsName,
		'Desc'       => $rsDesc,
		'OID'        => $rsOID,
		'OIDEnd'     => $rsOIDEnd,
		'AdmStatus'  => $refhSNMPResultRSAdmStatus->{"$oid_RServerAdmStatus{'oid'}$rsOIDEnd"},
		'OperStatus' => $refhSNMPResultRSOperStatus->{"$oid_RServerOperStatus{'oid'}$rsOIDEnd"},
		'StateDesc'  => $refhSNMPResultRSStateDesc->{"$oid_RServerStateDesc{'oid'}$rsOIDEnd"},
		'NCons'      => $refhSNMPResultRSNCons->{"$oid_RServerNumOfCons{'oid'}$rsOIDEnd"},
		'TCons'      => $refhSNMPResultRSTCons->{"$oid_RServerTotalCons{'oid'}$rsOIDEnd"},
        });
    }

    return 0;
}

# ------------------------------------------------------------------------------
# Get_SFarms
# ------------------------------------------------------------------------------
# Description: gather details about the ServerFarms in this context
# ------------------------------------------------------------------------------
sub Get_SFarms {
    my $refhOIDSFarms = shift;
    my %hOptions = ();

    # get all RServers in all ServerFarms
    eval {
        my %hOptions = ( %hOptionDefaults, (oids => ["$refhOIDSFarms->{'oid'}"]));
        $refhSNMPResultSFs = GetTableDataWithSnmp (\%hOptions);
    };
    ExitPlugin("UNKNOWN", add_oid_details($@->message(), [ $refhOIDSFarms ])) if ($@);

    # iterate over all RServers in all ServerFarms (by using AdminStatus OIDs)
    while (my ($sfOID, $sfDesc) = each(%$refhSNMPResultSFs) ) {
	my @sfrsOIDEnds = split($refhOIDSFarms->{'oid'}, $sfOID, 2);
	my $sfrsOIDEnd = $sfrsOIDEnds[1];
	my $sfOIDEnd;
	my $sfName;
	# iterate over all global RServers and find the one corresponding to
	# this particular RServer in the ServerFarm
	foreach (@RServers) {
	    # $rsOIDEnd -> the OID ending for current global RServer
	    # $sfrsOIDEnd -> the OID ending for current RServer in ServerFarm
	    (my $rsOIDEnd = $_->{'OIDEnd'}) =~ s/^\.1//;
	    # sfrsOIDEnd needs to end with $rsOIDEnd and a port number (\.d+)
	    if ($sfrsOIDEnd =~ /\Q$rsOIDEnd\E\.\d+$/) {
		my @sfOIDEnds = split($rsOIDEnd, $sfrsOIDEnd);
		$sfOIDEnd = $sfOIDEnds[0];

		# translate ServerFarmRServerAdminStatus OID using MIB
		my $sfrsOIDTrans = TranslateOID("$oid_SFarmRSvrAStatus{'oid'}$sfrsOIDEnd");
		# fetch ServerFarm name from translated OID name
		($sfName = $sfrsOIDTrans) =~ s/^CISCO-ENHANCED-SLB-MIB::cesServerFarmRserverAdminStatus\.1\."([^"]+)"\.".*$/$1/;
	    }
	}

	# add ServerFarm to @SFarms array only if it's not already there
	my $sfFlag = 0;
	foreach(@SFarms) {
	    if ($sfName =~ /\Q$_->{'Name'}\E/) { $sfFlag = 1 };
	}
	if ($sfFlag == 0) {
	    push(@SFarms, {
		'OID'        => "$oid_SFarmRSvrAStatus{'oid'}$sfOIDEnd",
		'OIDEnd'     => $sfOIDEnd,
		'Name'       => $sfName,
		'RServers'   => {},
	    });
	}
	$sfFlag = 1;
    }

    # iterate over all ServerFarms and get all RServers for each
    foreach (@SFarms) {
	my @SFRServers;

	# get all RServers in the ServerFarm (by using AdminStatus OIDs)
	eval {
	    my %hOptions = ( %hOptionDefaults, (oids => [ "$oid_SFarmRSvrAStatus{'oid'}$_->{'OIDEnd'}" ]));
            $refhSNMPResultSFRSs = GetTableDataWithSnmp (\%hOptions);
	};
	ExitPlugin("UNKNOWN", add_oid_details($@->message(), [ %oid_SFarmRSvrAStatus ])) if ($@);

	# iterate over all RServers in the ServerFarm (by using AdminStatus OIDs)
	while (my ($sfrsOID, $sfrsAdmStatus) = each(%$refhSNMPResultSFRSs) ) {
	    my @sfrsOIDEnds = split("$oid_SFarmRSvrAStatus{'oid'}$_->{'OIDEnd'}", $sfrsOID, 2);
	    my $sfrsOIDEnd = $sfrsOIDEnds[1];
	    %hOptions = ( %hOptionDefaults, (oids => [ "$oid_SFarmRSvrAStatus{'oid'}$_->{'OIDEnd'}$sfrsOIDEnd" ]));
	    my $refhSNMPResultSFAdmStatus = GetDataWithSnmp(\%hOptions);
	    %hOptions = ( %hOptionDefaults, (oids => [ "$oid_SFarmRSvrOStatus{'oid'}$_->{'OIDEnd'}$sfrsOIDEnd" ]));
	    my $refhSNMPResultSFOperStatus = GetDataWithSnmp(\%hOptions);
	    %hOptions = ( %hOptionDefaults, (oids => [ "$oid_SFarmRSvrStDesc{'oid'}$_->{'OIDEnd'}$sfrsOIDEnd" ]));
	    my $refhSNMPResultSFStateDesc = GetDataWithSnmp(\%hOptions);
	    %hOptions = ( %hOptionDefaults, (oids => [ "$oid_SFarmRSvrTotCons{'oid'}$_->{'OIDEnd'}$sfrsOIDEnd" ]));
	    my $refhSNMPResultSFNCons = GetDataWithSnmp(\%hOptions);
	    %hOptions = ( %hOptionDefaults, (oids => [ "$oid_SFarmRSvrNumCons{'oid'}$_->{'OIDEnd'}$sfrsOIDEnd" ]));
	    my $refhSNMPResultSFTCons = GetDataWithSnmp(\%hOptions);

	    # translate ServerFarmRServerAdminStatus OID using MIB
	    my $sfrsOIDTrans = TranslateOID("$oid_SFarmRSvrAStatus{'oid'}$_->{'OIDEnd'}$sfrsOIDEnd");
	    # fetch ServerFarm name from translated OID name
	    (my $sfrsName = $sfrsOIDTrans) =~ s/^CISCO-ENHANCED-SLB-MIB::cesServerFarmRserverAdminStatus\.1\."$_->{'Name'}"\."([^"]+)"\..*$/$1/;
	    push(@SFRServers, {
		'Name'       => $sfrsName,
		'OID'        => $sfrsOID,
		'OIDEnd'     => $sfrsOIDEnd,
		'AdmStatus'  => $refhSNMPResultSFAdmStatus->{"$oid_SFarmRSvrAStatus{'oid'}$_->{'OIDEnd'}$sfrsOIDEnd"},
		'OperStatus' => $refhSNMPResultSFOperStatus->{"$oid_SFarmRSvrOStatus{'oid'}$_->{'OIDEnd'}$sfrsOIDEnd"},
		'StateDesc'  => $refhSNMPResultSFStateDesc->{"$oid_SFarmRSvrStDesc{'oid'}$_->{'OIDEnd'}$sfrsOIDEnd"},
		'TCons'      => $refhSNMPResultSFNCons->{"$oid_SFarmRSvrTotCons{'oid'}$_->{'OIDEnd'}$sfrsOIDEnd"},
		'NCons'      => $refhSNMPResultSFTCons->{"$oid_SFarmRSvrNumCons{'oid'}$_->{'OIDEnd'}$sfrsOIDEnd"},
	    });
	    $_->{'RServers'} = \@SFRServers;
	}
    }

    return 0;
}

# ------------------------------------------------------------------------------
# Get_VServers
# ------------------------------------------------------------------------------
# Description: gather details about the VServers in this context
# ------------------------------------------------------------------------------
sub Get_VServers {
    my $refhOIDVServers = shift;
    my %hOptions = ();

    # get all VServers
    eval {
        my %hOptions = ( %hOptionDefaults, (oids => ["$refhOIDVServers->{'oid'}"]));
        $refhSNMPResultVSs = GetTableDataWithSnmp (\%hOptions);
    };
    ExitPlugin("UNKNOWN", add_oid_details($@->message(), [ $refhOIDVServers ])) if ($@);

    while (my ($vsOID, $vsClassMap) = each(%$refhSNMPResultVSs) ) {
	my @vsOIDEnds = split($refhOIDVServers->{'oid'}, $vsOID, 2);
        my $vsOIDEnd = $vsOIDEnds[1];
	%hOptions = ( %hOptionDefaults, (oids => [ "$oid_VServerState{'oid'}$vsOIDEnd" ]));
	my $refhSNMPResultVSState = GetDataWithSnmp(\%hOptions);
	%hOptions = ( %hOptionDefaults, (oids => [ "$oid_VServerStateDesc{'oid'}$vsOIDEnd" ]));
	my $refhSNMPResultVSStateDesc = GetDataWithSnmp(\%hOptions);
	%hOptions = ( %hOptionDefaults, (oids => [ "$oid_VServerNumOfCons{'oid'}$vsOIDEnd" ]));
	my $refhSNMPResultVSNCons = GetDataWithSnmp(\%hOptions);
	%hOptions = ( %hOptionDefaults, (oids => [ "$oid_VServerTotalCons{'oid'}$vsOIDEnd" ]));
	my $refhSNMPResultVSTCons = GetDataWithSnmp(\%hOptions);
        push(@VServers, {
		'ClassMap'  => $vsClassMap,
		'OID'       => $vsOID,
		'OIDEnd'    => $vsOIDEnd,
		'State'     => $refhSNMPResultVSState->{"$oid_VServerState{'oid'}$vsOIDEnd"},
		'StateDesc' => $refhSNMPResultVSStateDesc->{"$oid_VServerStateDesc{'oid'}$vsOIDEnd"},
		'NCons'     => $refhSNMPResultVSNCons->{"$oid_VServerNumOfCons{'oid'}$vsOIDEnd"},
		'TCons'     => $refhSNMPResultVSTCons->{"$oid_VServerTotalCons{'oid'}$vsOIDEnd"},
        });
    }

    return 0;
}

# ------------------------------------------------------------------------------
# Check_RServers
# ------------------------------------------------------------------------------
# Description: check status of RServer status
# ------------------------------------------------------------------------------
sub Check_RServer {
    my $rs_check = shift;
    # OperStatus != inService
    if ($rs_check->{'OperStatus'} != 2) {
	# AdminStatus != outOfService
	if ($rs_check->{'AdmStatus'} != 2) {
	    $pout .= " Warning: OperStatus ".$rserver_oper_status{$rs_check->{'OperStatus'}}." and AdminStatus ".$rserver_adm_status{$rs_check->{'AdmStatus'}};
	}
	else {
	    $pout .= " ".$rserver_oper_status{$rs_check->{'OperStatus'}};
	}
    }
    else {
	$pout .= " ".$rserver_oper_status{$rs_check->{'OperStatus'}};
	$rs_active++;
    }
}

# ------------------------------------------------------------------------------
# TranslateOID
# ------------------------------------------------------------------------------
# Description: translate a SNMP OID using CISCO-ENHANCED-SLB-MIB
# ------------------------------------------------------------------------------
sub TranslateOID {
    my $snmpTranslateCommand = "snmptranslate";
    my $snmpTranslateOptions = "-M-/usr/share/mibs/cisco-inet -m CISCO-ENHANCED-SLB-MIB";
    my $snmpTranslateOID = shift;

    my $snmpTranslatedName = `$snmpTranslateCommand $snmpTranslateOptions $snmpTranslateOID`;
    chomp($snmpTranslatedName);
    return $snmpTranslatedName;
}

# ------------------------------------------------------------------------------
# ExitPlugin
# ------------------------------------------------------------------------------
# Description: print correct output text and exit this plugin now
# ------------------------------------------------------------------------------
sub ExitPlugin {
    my $code = shift;
    my $output = shift;
    chomp($output);

    # Append errorcode string prefix to text
    if (! defined $ERRORS{$code}) {
	$code = "UNKNOWN";
    }
    $output = $code." - ".$output;

    print $output;

    # Append status to perfdata
    $perfdata .= "Status=".$ERRORS{$code};
    # Append performance data
    print " | $perfdata" if ($perfdata);
    print "\n";

    exit $ERRORS{$code};
}


# ------------------------------------------------------------------------------
# add_oid_details
# ------------------------------------------------------------------------------
# Description: check a string and add the object name and the mib corresponding
# to an oid if this one is found
# ------------------------------------------------------------------------------
sub add_oid_details {
    my $message = shift;
    my $refaohOID = shift;
    foreach my $refhOID (@{$refaohOID}) {
        $message =~ s/(\"+$refhOID->{'oid'}\"+)/$1 ($refhOID->{'name'}, mib $refhOID->{'mib'})/g;
    }
    return $message;
}

