/*
 *  Entangle: Tethered Camera Control & Capture
 *
 *  Copyright (C) 2009-2018 Daniel P. Berrangé
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 3 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#include <stdio.h>
#include <string.h>

#include <glib.h>

#include "entangle-video.h"

#include "entangle-debug.h"

/**
 * SECTION:entangle-video
 * @Short_description: a video media file
 * @Title: EntangleVideo
 *
 * The #EntangleVideo object presents a media file containing a
 * video stream.
 */

struct _EntangleVideo
{
    EntangleMedia parent;
    GstBaseSrc *source;
};

G_DEFINE_TYPE(EntangleVideo, entangle_video, ENTANGLE_TYPE_MEDIA);

enum
{
    PROP_0,
    PROP_SOURCE,
};

static void
entangle_video_get_property(GObject *object,
                            guint prop_id,
                            GValue *value,
                            GParamSpec *pspec)
{
    EntangleVideo *video = ENTANGLE_VIDEO(object);

    switch (prop_id) {
    case PROP_SOURCE:
        g_value_set_object(value, video->source);
        break;

    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
    }
}

static void
entangle_video_set_property(GObject *object,
                            guint prop_id,
                            const GValue *value,
                            GParamSpec *pspec)
{
    EntangleVideo *video = ENTANGLE_VIDEO(object);

    switch (prop_id) {
    case PROP_SOURCE:
        if (video->source)
            g_object_unref(video->source);
        video->source = g_value_get_object(value);
        if (video->source)
            g_object_ref(video->source);
        break;

    default:
        G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
    }
}

static void
entangle_video_finalize(GObject *object)
{
    EntangleVideo *video = ENTANGLE_VIDEO(object);

    ENTANGLE_DEBUG("Finalize video %p", object);

    if (video->source)
        g_object_unref(video->source);

    G_OBJECT_CLASS(entangle_video_parent_class)->finalize(object);
}

static void
entangle_video_class_init(EntangleVideoClass *klass)
{
    GObjectClass *object_class = G_OBJECT_CLASS(klass);

    object_class->finalize = entangle_video_finalize;
    object_class->get_property = entangle_video_get_property;
    object_class->set_property = entangle_video_set_property;

    g_object_class_install_property(
        object_class, PROP_SOURCE,
        g_param_spec_object("source", "Video source", "Video source",
                            GST_TYPE_BASE_SRC,
                            G_PARAM_READWRITE | G_PARAM_STATIC_NAME |
                                G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB));
}

/**
 * entangle_video_new_file:
 * @filename: (transfer none): the filename of the video
 *
 * Create a new video instance for the video stored in
 * @filename.
 *
 * Returns: (transfer full): the new video
 */
EntangleVideo *
entangle_video_new_file(const char *filename)
{
    return ENTANGLE_VIDEO(
        g_object_new(ENTANGLE_TYPE_VIDEO, "filename", filename, NULL));
}

EntangleVideo *
entangle_video_new_source(GstBaseSrc *source)
{
    return ENTANGLE_VIDEO(
        g_object_new(ENTANGLE_TYPE_VIDEO, "source", source, NULL));
}

static void
entangle_video_init(EntangleVideo *video G_GNUC_UNUSED)
{}

/**
 * entangle_video_get_source:
 * @video: (transfer none): the video to get the source for
 *
 * Get the live video source element
 *
 * Returns: (transfer none): the video source
 */
GstBaseSrc *
entangle_video_get_source(EntangleVideo *video)
{
    return video->source;
}

/**
 * entangle_video_set_source:
 * @video: (transfer none): the video to set the source for
 * @source: (transfer none): the new video source or NULL
 *
 * Set the live video source element
 */
void
entangle_video_set_source(EntangleVideo *video, GstBaseSrc *source)
{
    if (video->source)
        g_object_unref(source);
    video->source = source;
    if (video->source)
        g_object_ref(source);
}

/*
 * Local variables:
 *  c-indent-level: 4
 *  c-basic-offset: 4
 *  indent-tabs-mode: nil
 *  tab-width: 8
 * End:
 */
