/*---------------------------------------------------------------------------*\

    FILE....: MULAW.C
    TYPE....: C Functions
    AUTHOR..: David Rowe
    DATE....: 16/10/97

    Mu-law compression functions.

         Voicetronix Voice Processing Board (VPB) Software
         Copyright (C) 1999-2007 Voicetronix www.voicetronix.com.au

         This library is free software; you can redistribute it and/or
         modify it under the terms of the GNU Lesser General Public
         License as published by the Free Software Foundation; either
         version 2.1 of the License, or (at your option) any later version.

         This library is distributed in the hope that it will be useful,
         but WITHOUT ANY WARRANTY; without even the implied warranty of
         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
         Lesser General Public License for more details.

         You should have received a copy of the GNU Lesser General Public
         License along with this library; if not, write to the Free Software
         Foundation, Inc., 51 Franklin St, Fifth Floor, Boston,
         MA  02110-1301  USA

\*---------------------------------------------------------------------------*/

#include <assert.h>
#include <stdint.h>

#include "alawmulaw.h"

/* mu-law constants */

#define	LSIGN	0x8000		/* linear sign bit 15 = 1 if negative	   */
#define	MUSIGN	0x80		/* mu-law sign bit 7 = 1 if negative	   */
#define	MUMAX	0x1fff		/* max mu-law value in linear form	   */
#define	MUMAG	0x7f		/* mu-law magnitude bits 		   */
#define	BIAS	33		/* converts seg end points to powers of 2  */
#define	LINEAR	0xf		/* masks of linear portion of magnitude	   */

static short mulaw_to_linear_lut[256];
static char linear_to_mulaw_lut[65536];

static void init_mulaw_to_linear_lut(void);
static void init_linear_to_mulaw_lut(void);

/*--------------------------------------------------------------------------*\

	FUNCTION.: init_mulaw_luts
	AUTHORS..: Mark Mickan
	DATE.....: 29 Jan 2007

	Call the necessary lookup table initialisation functions.  This
	function should be called in initialisation code, before either
	mulaw_encode or mulaw_decode are called.

\*--------------------------------------------------------------------------*/

void init_mulaw_luts(void)
{
    init_mulaw_to_linear_lut();
    init_linear_to_mulaw_lut();
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: init_mulaw_to_linear_lut
	AUTHORS..: Mark Mickan
	DATE.....: 29 Jan 2007

	Calculate the linear equivalent of all possible mu-law samples.
	Heavily based on David Rowe's original mulaw_decode function.
	The linear samples are in Q15 format.

	Reference: TI databook
		   "Theory, Alogorithms, and Implementations Vol 1", p171

\*--------------------------------------------------------------------------*/

static void init_mulaw_to_linear_lut(void)
{
    unsigned long  acc,p,s,q;
    int    i;

    for(i=0; i<256; i++) {
	acc = ~i & 0xff;

	/* extract sign */

	if (acc & MUSIGN)
	    p = LSIGN;
	else
	    p = 0;
	acc &= MUMAG;

	/* extract q and s */

	q = acc & 0xf;
	s = acc >> 4;

	/* remove bias and form linear sample */

	acc = 2*q+BIAS;
	acc <<= s;
	acc -= BIAS;
	acc <<= MULAW_SCALE;
	if (p)
	    mulaw_to_linear_lut[i] = -acc;
	else
	    mulaw_to_linear_lut[i] = acc;
    }
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: init_linear_to_mulaw_lut
	AUTHOR...: Mark Mickan
	DATE.....: 29 Jan 2007

	Calculate the mu-law equivalent of all possible linear samples
	and store them in a lookup table.  The linear samples are
       	assumed to be in Q15 format.  Heavily based on David Rowe's
	original mulaw_encode function.

	Reference: TI databook
		   "Theory, Alogorithms, and Implementations Vol 1", p171

	Mu-law format: psssqqqq

		       p is the sign
		       s is the segment
		       q if the quantisation bin number

\*--------------------------------------------------------------------------*/

static void init_linear_to_mulaw_lut(void)
{
    unsigned short y,mag;
    unsigned short p,s,q,b;
    unsigned long  acc;
    int    i;

    for(i=-32768; i<32768; i++) {
	acc = i >> MULAW_SCALE;

	/* separate sign and bias */

	if (acc & LSIGN) {
	    p = MUSIGN;
	    mag = -acc;
	}
	else {
	    p = 0;
	    mag = acc;
	}
	mag += BIAS;

	/* encode magnitude */

	if (mag > MUMAX)
	    y = p + MUMAG;
	else {

	    /* determine left most bit and therefore segment */

	    b = 0;
	    acc = mag >> 1;
	    while(acc) {
		b++;
		acc >>= 1;
	    }
	    s = b - 5;

	    /* extract quantisation bin */

	    q = mag >> (b-4);
	    q &= LINEAR;
	    y = p + (s<<4) + q;
	}

	linear_to_mulaw_lut[i+32768] = ~y & 0xff;
    }
}


/*--------------------------------------------------------------------------*\

	FUNCTION.: mulaw_encode
	AUTHOR...: David Rowe
	DATE.....: 14/10/97

	Encodes a vector of linear samples to mu-law equivalents.  The linear
	samples are assumed to be in Q15 format.

\*--------------------------------------------------------------------------*/

void mulaw_encode(char mulaw[], const short linear[], unsigned short sz)
/*  char   mulaw[];	mulaw encoded samples                           */
/*  word   linear[];	16 bit input samples	                        */
/*  unsigned short sz;	number of sample in buffers			*/
{
    int    i;

    for(i=0; i<sz; ++i) {
	mulaw[i] = linear_to_mulaw_lut[linear[i]+32768];
    }
}

/*--------------------------------------------------------------------------*\

	FUNCTION.: mulaw_decode
	AUTHOR...: David Rowe
	DATE.....: 14/10/97
	AUTHOR...: Mark Mickan (modified to use lookup table)
	DATE.....: 29 Jan 2007

	Decodes a vector of mu-law samples to linear equivalents.  The linear
	samples are normalised to Q15.

\*--------------------------------------------------------------------------*/

void mulaw_decode(short linear[], const unsigned char mulaw[], unsigned short sz)
/*  word   linear[];	16 bit output samples           		*/
/*  char   mulaw[];	mulaw encoded samples                           */
/*  unsigned short sz;	number of sample in buffers			*/
{
    int    i;

    for(i=0; i<sz; ++i) {
	linear[i] = mulaw_to_linear_lut[mulaw[i]];
    }

}

