/* NVTV NVidia internal encoder data -- Dirk Thierbach <dthierbach@gmx.de>
 *
 * This file is part of nvtv, a tool for tv-output on NVidia cards.
 * 
 * nvtv is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * nvtv is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 * $Id: data_nx.c,v 1.3 2003/06/27 16:02:01 dthierbach Exp $
 *
 * Contents:
 *
 * Data tables and routines for the NVidia internal encoder chip.
 *
 */

#include <stddef.h>
#include <stdlib.h>
#include <math.h>

#include "data_nx.h"
#include "data_nv.h"

/* -------- NX -------- 60 Hz -------- */

TVNxImgRegs nx_60_normal_a = { /* NTSC 640x480 */
  scaler_h : 720,
  scaler_v : 240,
  overscan : 896,
  vip1     : 1762,
  vip2     : 6431,
};

TVNxImgRegs nx_60_normal_b = { /* NTSC 800x600 */
  scaler_h : 720,
  scaler_v : 240,
  overscan : 896,
  vip1     : 1188,
  vip2     : 7005,
}; 

TVNxImgRegs nx_60_normal_c = { /* NTSC 1024x768 */
  scaler_h : 720,
  scaler_v : 240,
  overscan : 896,
  vip1     : 369,
  vip2     : 7823,
}; 

/* -------- NX -------- 50 Hz -------- */

TVNxImgRegs nx_50_normal_a = { /* PAL 640x480 */
  scaler_h : 720,
  scaler_v : 288,
  overscan : 896,
  vip1     : 3994,
  vip2     : 103,
};

TVNxImgRegs nx_50_normal_b = { /* PAL 800x600 */
  scaler_h : 720,
  scaler_v : 288,
  overscan : 896,
  vip1     : 6534,
  vip2     : 1659,
};

TVNxImgRegs nx_50_normal_c = { /* PAL 1024x768 */
  scaler_h : 720,
  scaler_v : 288,
  overscan : 896,
  vip1     : 7209,
  vip2     : 983,
}; 

/* -------- NX -------- CRT -------- */

/* -------- System -------- */

/* SVideo: CONN-3, !CONN-5 */

TVNxIndRegs nx_pal = {
  fsub     : 705268427,
  nxreg04  : 0,
  nxreg08  : 126,
  nxreg09  : 64,
  nxreg0a  : 138,
  nxreg0b  : 50,
  nxreg0c  : 35,
  nxreg0e  : 251, 
  nxreg10  : 251,
  nxreg17  : 625,
  nxreg1e  : 800,
  nxreg20  : 166,
  nxreg22  : 166,
  nxreg29  : 26,
  nxreg2c  : 24,
  nxreg31  : 1440, /* 2x720 */
  nxreg35  : 16,
  nxreg3c  : 251,  /* = reg10 */
  nxsys12  : 0,
  flags	   : NX_FLAG_SYS_0 | NX_FLAG_SYS_3 | NX_FLAG_SYS_6,
};

TVNxIndRegs nx_ntsc = {
  fsub     : 569408543,
  nxreg04  : 0,
  nxreg08  : 126,
  nxreg09  : 68,
  nxreg0a  : 118,
  nxreg0b  : 67,
  nxreg0c  : 0,
  nxreg0e  : 282,
  nxreg10  : 240,
  nxreg17  : 525,
  nxreg1e  : 800,
  nxreg20  : 156,
  nxreg22  : 156,
  nxreg29  : 22,
  nxreg2c  : 32,
  nxreg31  : 1440, /* 2x720 */
  nxreg35  : 16,
  nxreg3c  : 240,  /* = reg10 */
  nxsys12  : 2,
  flags    : 0, /* !SYS-0 !SYS-3 !SYS-6 */
};


/* different from ntsc: 0e 20 22 */
TVNxIndRegs nx_ntsc_j = {
  fsub     : 569408543,
  nxreg04  : 0,
  nxreg08  : 126,
  nxreg09  : 68,
  nxreg0a  : 118,
  nxreg0b  : 67,
  nxreg0c  : 0,
  nxreg0e  : 240, /* different */
  nxreg10  : 240,
  nxreg17  : 525,
  nxreg1e  : 800,
  nxreg20  : 166, /* different */
  nxreg22  : 166, /* different */
  nxreg29  : 22,
  nxreg2c  : 32,
  nxreg31  : 1440, /* 2x720 */
  nxreg35  : 16,
  nxreg3c  : 240,  /* = reg10 */
  nxsys12  : 2,
  flags    : 0, /* !SYS-0 !SYS-3 !SYS-6 */
};

TVNxIndRegs nx_pal_m = { /* unchecked */
  fsub     : 568782819,
  nxreg04  : 0,
  nxreg08  : 126,
  nxreg09  : 68,
  nxreg0a  : 118,
  nxreg0b  : 47,
  nxreg0c  : 33,
  nxreg0e  : 282,
  nxreg10  : 240,
  nxreg17  : 625,
  nxreg1e  : 800,
  nxreg20  : 156,
  nxreg22  : 156,
  nxreg29  : 18,
  nxreg2c  : 32,
  nxreg31  : 1440, /* 2x720 */
  nxreg35  : 16,
  nxreg3c  : 240,  /* = reg10 */
  nxsys12  : 0,
  flags    : NX_FLAG_SYS_6, /* !SYS-0 !SYS-3 SYS-6 */
};

TVNxIndRegs nx_pal_n = {
  fsub     : 705268427,
  nxreg04  : 0,
  nxreg08  : 126,
  nxreg09  : 64,
  nxreg0a  : 138,
  nxreg0b  : 47,
  nxreg0c  : 33,
  nxreg0e  : 282, 
  nxreg10  : 240,
  nxreg17  : 625,
  nxreg1e  : 800,
  nxreg20  : 156,
  nxreg22  : 156,
  nxreg29  : 26,
  nxreg2c  : 24,
  nxreg31  : 1440, /* 2x720 */
  nxreg35  : 16,
  nxreg3c  : 240,  /* = reg10 */
  nxsys12  : 0,
  flags	   :  NX_FLAG_SYS_6, /* !SYS-0 !SYS-3 SYS-6 */
};

TVNxIndRegs nx_pal_nc = {
  fsub     : 569807942,
  nxreg04  : 0,
  nxreg08  : 126,
  nxreg09  : 68,
  nxreg0a  : 138,
  nxreg0b  : 50,
  nxreg0c  : 35,
  nxreg0e  : 251, 
  nxreg10  : 251,
  nxreg17  : 625,
  nxreg1e  : 800,
  nxreg20  : 166,
  nxreg22  : 166,
  nxreg29  : 26,
  nxreg2c  : 24,
  nxreg31  : 1440, /* 2x720 */
  nxreg35  : 16,
  nxreg3c  : 251,  /* = reg10 */
  nxsys12  : 0,
  flags	   : NX_FLAG_SYS_0 | NX_FLAG_SYS_3 | NX_FLAG_SYS_6,
};

/* --------  -------- */

void 
data_init_nx (TVSystem system, TVNxRegs *r) 
{
  int i, j;

  if (!r) return;
  r->muxa = 2; /* SVideo */
  r->muxb = 1; /* SVideo */
  r->muxc = 0; /* SVideo */
  r->filter = 1; /* depends on img ... */
  r->flags1 = NX_FLAG1_SVIDEO;
  switch (system) {
    case TV_SYSTEM_NTSC: /* additional fsci values for NDC (no dot crawl) */
      r->ind = nx_ntsc;
      break;
    case TV_SYSTEM_NTSC_J:
      r->ind = nx_ntsc_j;
      break;
    case TV_SYSTEM_NONE:  /* to have a default */
    case TV_SYSTEM_SECAM: /* makes most sense */
    case TV_SYSTEM_PAL:
    case TV_SYSTEM_PAL_X:
      r->ind = nx_pal;
      break;
    case TV_SYSTEM_PAL_N:  /* PAL-N has different fsci values */
      r->ind = nx_pal_n;
      break;
    case TV_SYSTEM_PAL_NC: /* guess */
      r->ind = nx_pal_nc;
      break;
    case TV_SYSTEM_PAL_M:   /* needs different fsci values */
    case TV_SYSTEM_PAL_M60: /* guess */
      r->ind = nx_pal_m;
      break;
    case TV_SYSTEM_PAL_60: /* modify PAL_M */
      r->ind = nx_pal;     /* FIXME for now */
      break;
  }
  r->ind.flags &= ~(NX_FLAG_CONN_3 | NX_FLAG_CONN_5);
  r->ind.flags |= NX_FLAG_CONN_3; /* SVideo: CONN-3, !CONN-5 */
  for (i = 0; i <= 3; i++) {
    for (j = 0; j <= 7; j++) {
      r->filt_x1[i][j] = 0;
      r->filt_x2[i][j] = 0;
      r->filt_y[i][j] = 0;
    }
  }
}

void 
data_default_nx (TVSettings *s)
{
  s->tv_hoffset = s->mon_hoffset = 0;
  s->tv_voffset = s->mon_voffset = 0;
  s->brightness_sig = 0;
  s->contrast_sig = 0;
  s->contrast = 0;
  s->saturation_sig = 0;
  s->saturation = 0;
  s->flicker = 75;
  s->flicker_adapt = 0;
  s->luma_bandwidth = 100;
  s->chroma_bandwidth = 75;
  s->sharpness = 50;
  s->cross_color = 0;
  s->phase = 0;
  s->hue = 0;
  /* FIXME flags */
}

void 
data_clamp_nx (TVSettings *s, TVRegs *r)
{
}

void 
data_setup_nx (TVSettings *s, TVRegs *r)
{
}

void
data_calc_vip_nx (TVRegs *r)
{
  int low, high;

  low = (2048 * r->crtc.nv.VDisplay + 1024) / 
    ((r->enc.nx.img.scaler_v * r->enc.nx.img.overscan + 512) / 1024);
  high = (low / 4096) * 4096;
  if (high < low) high += 4096;
  if (r->enc.nx.ind.flags & NX_FLAG_SYS_0) {
    r->enc.nx.img.vip1 = (high + low) / 2;
    r->enc.nx.img.vip2 = (high - low) / 2;
  } else {
    r->enc.nx.img.vip1 = (high - low) / 2;
    r->enc.nx.img.vip2 = (high + low) / 2;
  }
}

static int filter_coeff[4][4][7] = {
  {{   63,   59,   46,   45,   46,   59,   63},
   {  -14,  -17,   -1,   22,   47,   76,   78},
   {  -21,  -25,  -28,  -27,  -28,  -25,  -21},
   {    9,    8,   -4,  -14,  -24,  -33,  -29}},

  {{   65,   65,   66,   66,   66,   65,   65},
   {    3,    9,   25,   33,   41,   56,   62},
   {   -6,  -20,  -36,  -36,  -36,  -20,   -6},
   {  -12,  -16,  -20,  -18,  -16,   -4,    6}},

  {{   64,   65,   65,   64,   65,   65,   64},
   {   16,   19,   28,   32,   37,   45,   49},
   {   27,   20,   13,   13,   13,   20,   27},
   {   -5,   -4,    2,    7,   11,   24,   32}},

  {{   63,   63,   63,   63,   63,   63,   63},
   {   25,   27,   30,   32,   33,   36,   38},
   {   67,   65,   63,   63,   63,   65,   67},
   {   18,   21,   28,   32,   35,   44,   49}}
};

void
data_calc_filter_nx (TVRegs *r)
{
}

void
data_complete_nx (TVRegs *r)
{
  data_calc_vip_nx (r);
  data_calc_filter_nx (r);
}
