# ==define manila::backend::generic
#
# ===Parameters
#
# [*driver_handles_share_servers*]
#  (required) Denotes whether the driver should handle the responsibility of
#  managing share servers. This must be set to false if the driver is to
#  operate without managing share servers.
#
# [*share_backend_name*]
#   (optional) Name of the backend in manila.conf that
#   these settings will reside in
#
# [*backend_availability_zone*]
#   (Optional) Availability zone for this share backend.
#   If not set, the storage_availability_zone option value
#   is used as the default for all backends.
#   Defaults to $::os_service_default.
#
# [*smb_template_config_path*]
#   (optional) Path to smb config.
#   Defaults to: $state_path/smb.conf
#
# [*volume_name_template*]
#   (optional) Volume name template.
#   Defaults to: manila-share-%s
#
# [*volume_snapshot_name_template*]
#   (optional) Volume snapshot name template.
#   Defaults to: manila-snapshot-%s
#
# [*share_mount_path*]
#   (optional) Parent path in service instance where shares will be mounted.
#   Defaults to: /shares
#
# [*max_time_to_create_volume*]
#   (optional) Maximum time to wait for creating cinder volume.
#   Defaults to: 180
#
# [*max_time_to_attach*]
#   (optional) Maximum time to wait for attaching cinder volume.
#   Defaults to: 120
#
# [*service_instance_smb_config_path*]
#   (optional) Path to smb config in service instance.
#   Defaults to: $share_mount_path/smb.conf
#
# [*share_volume_fstype*]
#   (optional) Filesystem type of the share volume.
#   Choices: 'ext4', 'ext3'
#   Defaults to: ext4
#
# [*share_helpers*]
#   (optional) Specify list of share export helpers.
#   Defaults to: ['CIFS=manila.share.drivers.helpers.CIFSHelperIPAccess',
#                 'NFS=manila.share.drivers.helpers.NFSHelper']
#
# [*cinder_volume_type*]
#   (optional) Name or id of cinder volume type which will be used for all
#   volumes created by driver.
#
# [*delete_share_server_with_last_share*]
#   (optional) With this option is set to True share server will be deleted
#   on deletion of last share.
#   Defaults to: False
#
# [*unmanage_remove_access_rules*]
#   (optional) If set to True, then manila will deny access and remove all
#   access rules on share unmanage. If set to False - nothing will be changed.
#   Defaults to: False
#
# [*automatic_share_server_cleanup*]
#   (optional) If set to True, then Manila will delete all share servers which
#   were unused more than specified time. If set to False, automatic deletion
#   of share servers will be disabled.
#   Defaults to: True
#
define manila::backend::generic (
  $driver_handles_share_servers,
  $share_backend_name                  = $name,
  $backend_availability_zone           = $::os_service_default,
  $smb_template_config_path            = '$state_path/smb.conf',
  $volume_name_template                = 'manila-share-%s',
  $volume_snapshot_name_template       = 'manila-snapshot-%s',
  $share_mount_path                    = '/shares',
  $max_time_to_create_volume           = 180,
  $max_time_to_attach                  = 120,
  $service_instance_smb_config_path    = '$share_mount_path/smb.conf',
  $share_volume_fstype                 = 'ext4',
  $share_helpers = ['CIFS=manila.share.drivers.helpers.CIFSHelperIPAccess',
                    'NFS=manila.share.drivers.helpers.NFSHelper'],
  $cinder_volume_type                  = undef,
  $delete_share_server_with_last_share = 'False',
  $unmanage_remove_access_rules        = 'False',
  $automatic_share_server_cleanup      = 'True',
) {

  include manila::deps

  $share_driver = 'manila.share.drivers.generic.GenericShareDriver'

  manila_config {
    "${name}/driver_handles_share_servers":        value => $driver_handles_share_servers;
    "${name}/share_backend_name":                  value => $share_backend_name;
    "${name}/backend_availability_zone":           value => $backend_availability_zone;
    "${name}/share_driver":                        value => $share_driver;
    "${name}/smb_template_config_path":            value => $smb_template_config_path;
    "${name}/volume_name_template":                value => $volume_name_template;
    "${name}/volume_snapshot_name_template":       value => $volume_snapshot_name_template;
    "${name}/share_mount_path":                    value => $share_mount_path;
    "${name}/max_time_to_create_volume":           value => $max_time_to_create_volume;
    "${name}/max_time_to_attach":                  value => $max_time_to_attach;
    "${name}/service_instance_smb_config_path":    value => $service_instance_smb_config_path;
    "${name}/share_volume_fstype":                 value => $share_volume_fstype;
    "${name}/share_helpers":                       value => join($share_helpers, ',');
    "${name}/cinder_volume_type":                  value => $cinder_volume_type;
    "${name}/delete_share_server_with_last_share": value => $delete_share_server_with_last_share;
    "${name}/unmanage_remove_access_rules":        value => $unmanage_remove_access_rules;
    "${name}/automatic_share_server_cleanup":      value => $automatic_share_server_cleanup;
  }
}
