% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/aanma.pdbs.R
\name{aanma.pdbs}
\alias{aanma.pdbs}
\title{Ensemble Normal Mode Analysis with All-Atom ENM}
\usage{
\method{aanma}{pdbs}(pdbs, fit = TRUE, full = FALSE, subspace = NULL,
  rm.gaps = TRUE, ligand = FALSE, outpath = NULL, gc.first = TRUE,
  ncore = NULL, ...)
}
\arguments{
\item{pdbs}{an \sQuote{pdbs} object as obtained from \code{\link{read.all}}.}

\item{fit}{logical, if TRUE C-alpha coordinate based superposition is 
performed prior to normal mode calculations.}

\item{full}{logical, if TRUE return the complete, full structure,
\sQuote{nma} objects.}

\item{subspace}{number of eigenvectors to store for further analysis.}

\item{rm.gaps}{logical, if TRUE obtain the hessian matrices for only
atoms in the aligned positions (non-gap positions in all aligned
structures). Thus, gap positions are removed from output.}

\item{ligand}{logical, if TRUE ligand molecules are also included in the 
calculation.}

\item{outpath}{character string specifing the output directory to
which the PDB structures should be written.}

\item{gc.first}{logical, if TRUE will call gc() first before mode calculation
for each structure. This is to avoid memory overload when 
\code{ncore > 1}.}

\item{ncore}{number of CPU cores used to do the calculation.}

\item{...}{additional arguments to \code{\link{aanma}}.}
}
\value{
Returns a list of \sQuote{nma} objects (\code{outmodes} is provided 
   and is not \sQuote{calpha}) or an \sQuote{enma} object with the following 
   components:
   \item{fluctuations }{ a numeric matrix containing aligned atomic
     fluctuations with one row per input structure. }
   \item{rmsip }{ a numeric matrix of pair wise RMSIP values (only the ten
     lowest frequency modes are included in the calculation). }
   \item{U.subspace }{ a three-dimensional array with aligned
     eigenvectors  (corresponding to the subspace defined by the first N
     non-trivial eigenvectors (\sQuote{U}) of the \sQuote{nma} object). }
   \item{L }{ numeric matrix containing the raw eigenvalues with one row
     per input structure. }
   \item{full.nma }{ a list with a \code{nma} object for each input
     structure (available only when \code{full=TRUE}). }
}
\description{
Perform normal mode analysis (NMA) on an ensemble of aligned protein
structures using all-atom elastic network model (aaENM).
}
\details{
This function builds elastic network model (ENM) using all heavy 
   atoms and performs subsequent normal mode analysis (NMA) on a set of 
   aligned protein structures obtained with function \code{\link{read.all}}.
   The main purpose is to automate ensemble normal mode analysis using 
   all-atom ENMs.

   By default, the effective Hessian for all C-alpha atoms is calculated 
   based on the Hessian built from all heavy atoms (including ligand atoms if 
   \code{ligand=TRUE}). Returned values include aligned mode vectors and 
   (when \code{full=TRUE}) a list containing the full \sQuote{nma} objects
   one per each structure. When \sQuote{rm.gaps=TRUE} the unaligned atoms 
   are ommited from output. With default arguments \sQuote{rmsip} provides 
   RMSIP values for all pairwise structures.

   When \code{outmodes} is provided and is not \sQuote{calpha} 
   (e.g. \sQuote{noh}. See \code{\link{aanma}} for more details), the 
   function simply returns a list of \sQuote{nma} objects, one per each 
   structure, and no aligned mode vector is returned. In this case, the 
   arguments \code{full}, \code{subspace}, and \code{rm.gaps} are ignored. 
   This is equivalent to a wrapper function repeatedly calling 
   \code{\link{aanma}}.
}
\examples{
\donttest{
  # Needs MUSCLE installed - testing excluded
  if(check.utility("muscle")) {

    ## Fetch PDB files and split to chain A only PDB files
    ids <- c("1a70_A", "1czp_A", "1frd_A", "1fxi_A", "1iue_A", "1pfd_A")
    files <- get.pdb(ids, split = TRUE, path = tempdir())
    
    ## Sequence Alignement
    aln <- pdbaln(files, outfile = tempfile())
    
    ## Read all pdb coordinates
    pdbs <- read.all(aln)

    ## Normal mode analysis on aligned data
    modes <- aanma(pdbs, rm.gaps=TRUE)
    
    ## Plot fluctuation data
    plot(modes, pdbs=pdbs)
    
    ## Cluster on Fluctuation similariy
    sip <- sip(modes)
    hc <- hclust(dist(sip))
    col <- cutree(hc, k=3)
    
    ## Plot fluctuation data
    plot(modes, pdbs=pdbs, col=col)
    
    ## RMSIP is pre-calculated
    heatmap(1-modes$rmsip)
    
    ## Bhattacharyya coefficient
    bc <- bhattacharyya(modes)
    heatmap(1-bc)

  }
}
}
\author{
Xin-Qiu Yao & Lars Skjaerven
}
\seealso{
For normal mode analysis on single structure PDB:
     \code{\link{aanma}}

     For conventional C-alpha based normal mode analysis:
     \code{\link{nma}}, \code{\link{nma.pdbs}}.

     For the analysis of the resulting \sQuote{eNMA} object:
     \code{\link{mktrj.enma}}, \code{\link{dccm.enma}},
     \code{\link{plot.enma}}, \code{\link{cov.enma}}.
   
     Similarity measures:
     \code{\link{sip}}, \code{\link{covsoverlap}},
     \code{\link{bhattacharyya}}, \code{\link{rmsip}}.
   
     Related functionality:
     \code{\link{read.all}}.
}

