/* SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * Copyright:
 *   2020      Evan Nemerson <evan@nemerson.com>
 *   2020      Himanshi Mathur <himanshi18037@iiitd.ac.in>
 *   2020      Hidayat Khan <huk2209@gmail.com>
 */

#define SIMDE_TEST_X86_AVX512_INSN xor

#include <test/x86/avx512/test-avx512.h>
#include <simde/x86/avx512/set.h>
#include <simde/x86/avx512/xor.h>

static int
test_simde_mm512_xor_ps (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int32_t a[16];
    const int32_t b[16];
    const int32_t r[16];
  } test_vec[] = {
    { {  INT32_C(  2140215653),  INT32_C(  1293320897), -INT32_C(  1520002534),  INT32_C(   294609697),  INT32_C(  1091317370),  INT32_C(   313266332), -INT32_C(  1143786207),  INT32_C(  1969865590),
         INT32_C(  1062829004), -INT32_C(   597668763),  INT32_C(  1683087224),  INT32_C(   739479593), -INT32_C(   468073407), -INT32_C(   573950189), -INT32_C(   974763361),  INT32_C(   527665652) },
      {  INT32_C(   816665849),  INT32_C(  1611972509), -INT32_C(  1280024725), -INT32_C(  1519923382), -INT32_C(  1677919607), -INT32_C(  1493407047),  INT32_C(  1584582304), -INT32_C(  1720041634),
         INT32_C(    70381333),  INT32_C(   746736592), -INT32_C(  1868567415),  INT32_C(  1303201364),  INT32_C(  2110232181), -INT32_C(  1733384071),  INT32_C(    69343053), -INT32_C(  1142043763) },
      {  INT32_C(  1329352092),  INT32_C(   755121500),  INT32_C(   382911857), -INT32_C(  1259819925), -INT32_C(   621751053), -INT32_C(  1269794267), -INT32_C(   442369151), -INT32_C(   334234584),
         INT32_C(   996709593), -INT32_C(   253622347), -INT32_C(   187812879),  INT32_C(  1639896701), -INT32_C(  1713483212),  INT32_C(  1164222314), -INT32_C(  1044095534), -INT32_C(  1533132167) } },
    { { -INT32_C(  1116734570),  INT32_C(  1613310165),  INT32_C(   813802444),  INT32_C(   429161548), -INT32_C(  1674708291),  INT32_C(  1572063288),  INT32_C(   653388597), -INT32_C(   887480686),
         INT32_C(  1388041778), -INT32_C(   190487029),  INT32_C(   304982209),  INT32_C(   851454399), -INT32_C(   666906362),  INT32_C(  1120951633),  INT32_C(  2076068325),  INT32_C(  1217737327) },
      { -INT32_C(   807130602),  INT32_C(   590835955),  INT32_C(   594477440), -INT32_C(   617573334),  INT32_C(   291751240),  INT32_C(  1236534164),  INT32_C(   789759175), -INT32_C(    74502962),
        -INT32_C(   134459010), -INT32_C(  1974206140), -INT32_C(  1593126896), -INT32_C(   699738723), -INT32_C(   399365675), -INT32_C(  1537291680),  INT32_C(   615597302), -INT32_C(  1491990022) },
      {  INT32_C(  1921765760),  INT32_C(  1126062118),  INT32_C(   334406220), -INT32_C(  1029379994), -INT32_C(  1924251659),  INT32_C(   336055212),  INT32_C(   165883890),  INT32_C(   815084124),
        -INT32_C(  1522038964),  INT32_C(  2129749839), -INT32_C(  1289258799), -INT32_C(   460653534),  INT32_C(   806222035), -INT32_C(   426862799),  INT32_C(  1594821907), -INT32_C(   276350059) } },
    { { -INT32_C(   289806681),  INT32_C(  1144433863),  INT32_C(  2078393105), -INT32_C(   621193477),  INT32_C(   197942373), -INT32_C(   778195138), -INT32_C(  1500149552), -INT32_C(   376058582),
         INT32_C(  1008015921), -INT32_C(   817846632), -INT32_C(  1418259563), -INT32_C(  1728720625), -INT32_C(   875417463), -INT32_C(   433179879), -INT32_C(   401326522), -INT32_C(   293254336) },
      { -INT32_C(  1611473168),  INT32_C(  1779296220),  INT32_C(  1625584338),  INT32_C(  1628081346),  INT32_C(  1516234414), -INT32_C(    67171629),  INT32_C(  1608454162), -INT32_C(   114353618),
        -INT32_C(   353092690),  INT32_C(  1984997008),  INT32_C(  1607474741), -INT32_C(  1472407712), -INT32_C(     3064222),  INT32_C(   279389725), -INT32_C(  1869192019),  INT32_C(   302689758) },
      {  INT32_C(  1900755031),  INT32_C(   775637787),  INT32_C(   453358019), -INT32_C(  1141646791),  INT32_C(  1368636107),  INT32_C(   711108077), -INT32_C(   112551742),  INT32_C(   280680196),
        -INT32_C(   689888865), -INT32_C(  1190133752), -INT32_C(   190378592),  INT32_C(   818480751),  INT32_C(   872615659), -INT32_C(   158788348),  INT32_C(  2021808875), -INT32_C(    57678178) } },
    { {  INT32_C(   419761061),  INT32_C(  1205817843), -INT32_C(  1808728463), -INT32_C(   270741600),  INT32_C(    62568967), -INT32_C(  1758665902), -INT32_C(  1620063715),  INT32_C(   975059798),
         INT32_C(   672464530), -INT32_C(  1296717020),  INT32_C(   968463109), -INT32_C(   943239776),  INT32_C(  1563835967),  INT32_C(  1537408671),  INT32_C(   771343793),  INT32_C(  1593811067) },
      { -INT32_C(  1519602963),  INT32_C(  1238941430), -INT32_C(   210338261), -INT32_C(   240423445), -INT32_C(  1881846659),  INT32_C(   210890163),  INT32_C(   374811810), -INT32_C(  1712882298),
         INT32_C(  1497831658),  INT32_C(  2098029338), -INT32_C(   368469787),  INT32_C(  2040566000), -INT32_C(  1929731099),  INT32_C(  1291609713), -INT32_C(  1096372677), -INT32_C(   147699721) },
      { -INT32_C(  1133921976),  INT32_C(   235400965),  INT32_C(  1732673626),  INT32_C(   511156811), -INT32_C(  1938819462), -INT32_C(  1682108703), -INT32_C(  1992759617), -INT32_C(  1543944496),
         INT32_C(  1901323896), -INT32_C(   809973698), -INT32_C(   743437344), -INT32_C(  1100489392), -INT32_C(   775119398),  INT32_C(   392075502), -INT32_C(  1822463094), -INT32_C(  1446123636) } },
    { { -INT32_C(  1891676286), -INT32_C(  1841226010),  INT32_C(  1540983227), -INT32_C(  1986737150),  INT32_C(   397242270),  INT32_C(   823916557), -INT32_C(  1551338568),  INT32_C(  1077412441),
         INT32_C(  1885334403), -INT32_C(  1567613993), -INT32_C(  1973232663),  INT32_C(  1561190391),  INT32_C(   194947553),  INT32_C(   332812599),  INT32_C(  1009120275),  INT32_C(  1926064119) },
      { -INT32_C(  1175618672), -INT32_C(  1361499621),  INT32_C(  2080692609),  INT32_C(    36764393), -INT32_C(   844078996),  INT32_C(    35128981), -INT32_C(  1577126054), -INT32_C(  1409287093),
         INT32_C(  1241153522),  INT32_C(  1667693632), -INT32_C(  2140278031), -INT32_C(   250677594),  INT32_C(   771501117),  INT32_C(  1049317354), -INT32_C(   683384889),  INT32_C(  1735623030) },
      {  INT32_C(   919745554),  INT32_C(  1016607997),  INT32_C(   668815418), -INT32_C(  1952172309), -INT32_C(   635629582),  INT32_C(   855899800),  INT32_C(    41321698), -INT32_C(   339214830),
         INT32_C(   967152753), -INT32_C(  1040783465),  INT32_C(   168765720), -INT32_C(  1409078959),  INT32_C(   643991004),  INT32_C(   761077469), -INT32_C(   345920556),  INT32_C(   364832385) } },
    { { -INT32_C(  1727546569), -INT32_C(  2058639326),  INT32_C(  1338749765),  INT32_C(   373465026), -INT32_C(   671124678), -INT32_C(  1919302723), -INT32_C(  1233004256), -INT32_C(  1559179697),
        -INT32_C(   107798480), -INT32_C(   385456720), -INT32_C(   898044456), -INT32_C(  1625696711), -INT32_C(   376937145), -INT32_C(  1132367764),  INT32_C(   902481945),  INT32_C(   792056806) },
      { -INT32_C(  1841227009),  INT32_C(  1539669343), -INT32_C(   709735776),  INT32_C(  1652239698), -INT32_C(  1711680384), -INT32_C(  1429694270),  INT32_C(   525789643),  INT32_C(  1259471626),
         INT32_C(  1193841178),  INT32_C(   812285822),  INT32_C(   876026405),  INT32_C(   332646634), -INT32_C(  1055303881),  INT32_C(  1713064637), -INT32_C(  1819389369),  INT32_C(   379883641) },
      {  INT32_C(   189178312), -INT32_C(   561064579), -INT32_C(  1703282203),  INT32_C(  1949938320),  INT32_C(  1309059002),  INT32_C(   659643263), -INT32_C(  1445515029), -INT32_C(   402469563),
        -INT32_C(  1094995414), -INT32_C(   647226674), -INT32_C(    28313603), -INT32_C(  1932910893),  INT32_C(   680604272), -INT32_C(   627426607), -INT32_C(  1505453474),  INT32_C(   965821343) } },
    { { -INT32_C(   349030303),  INT32_C(  1785046111), -INT32_C(  1042757457),  INT32_C(  1737927527),  INT32_C(  1161594549),  INT32_C(  1160192042), -INT32_C(   184434809), -INT32_C(  1720055340),
        -INT32_C(   339521427), -INT32_C(  1970209792), -INT32_C(  1248287430), -INT32_C(   360789525),  INT32_C(  1913349951),  INT32_C(   360975298),  INT32_C(  1185309231),  INT32_C(  1025588088) },
      { -INT32_C(   388091286),  INT32_C(  1475451470), -INT32_C(   582153313), -INT32_C(  1991214562), -INT32_C(  1091731760), -INT32_C(   868334684), -INT32_C(  1166550823), -INT32_C(   734386821),
        -INT32_C(   499790847), -INT32_C(  1911490173),  INT32_C(   998569952),  INT32_C(  1668173516),  INT32_C(   240093475),  INT32_C(  1396415836),  INT32_C(  1050460006), -INT32_C(  1184904202) },
      {  INT32_C(    65800715),  INT32_C(  1033125393),  INT32_C(   479576880), -INT32_C(   288960135), -INT32_C(    70125467), -INT32_C(  1994824818),  INT32_C(  1333159774),  INT32_C(  1296051375),
         INT32_C(   167156844),  INT32_C(    75498883), -INT32_C(  1910746406), -INT32_C(  1995405529),  INT32_C(  2084892188),  INT32_C(  1186961054),  INT32_C(  2017123657), -INT32_C(  2072055666) } },
    { {  INT32_C(  1950700306), -INT32_C(   237283605), -INT32_C(  1190591724),  INT32_C(     2981687), -INT32_C(   576818779), -INT32_C(    20979385),  INT32_C(   750065778), -INT32_C(   830997516),
        -INT32_C(   852723094),  INT32_C(  1566901338),  INT32_C(   353305803),  INT32_C(     6679193),  INT32_C(  1342823370), -INT32_C(  1377161447),  INT32_C(  1791982968),  INT32_C(   243243187) },
      { -INT32_C(   928073445), -INT32_C(    35749180), -INT32_C(   356403761), -INT32_C(   863827258),  INT32_C(   822300177),  INT32_C(  1598338091), -INT32_C(  1771112626),  INT32_C(   117434161),
         INT32_C(   676273718),  INT32_C(   693777629),  INT32_C(   320965188),  INT32_C(  1812143731), -INT32_C(  1549201693), -INT32_C(  1144932349), -INT32_C(  2083228063),  INT32_C(  1723727495) },
      { -INT32_C(  1125389815),  INT32_C(   201708591),  INT32_C(  1405662939), -INT32_C(   860981775), -INT32_C(   325238860), -INT32_C(  1577364116), -INT32_C(  1159991492), -INT32_C(   930603835),
        -INT32_C(   446471076),  INT32_C(  1950298247),  INT32_C(   103714447),  INT32_C(  1818686698), -INT32_C(   207563479),  INT32_C(   371955482), -INT32_C(   384103655),  INT32_C(  1757570612) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i a = simde_mm512_loadu_epi32(test_vec[i].a);
    simde__m512i b = simde_mm512_loadu_epi32(test_vec[i].b);
    simde__m512i r = simde_mm512_castps_si512(simde_mm512_xor_ps(simde_mm512_castsi512_ps(a), simde_mm512_castsi512_ps(b)));
    simde_test_x86_assert_equal_i32x16(r, simde_mm512_loadu_epi32(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm512_mask_xor_ps (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const int32_t src[16];
    const simde__mmask16 k;
    const int32_t a[16];
    const int32_t b[16];
    const int32_t r[16];
  } test_vec[] = {
    { {  INT32_C(  1881296091),  INT32_C(   418875182),  INT32_C(   134011108),  INT32_C(   720570282), -INT32_C(  1853596235), -INT32_C(   496598853), -INT32_C(  1204688658),  INT32_C(  1068536932),
         INT32_C(   246469344),  INT32_C(  1026008921),  INT32_C(   692331391), -INT32_C(   548194518),  INT32_C(  1349638293),  INT32_C(  1177737048),  INT32_C(   603874239), -INT32_C(   681332745) },
      UINT16_C( 4993),
      {  INT32_C(   230349798),  INT32_C(  1563441432), -INT32_C(  1231791517), -INT32_C(  1433523911), -INT32_C(  1417550215),  INT32_C(   722419756), -INT32_C(   942012828), -INT32_C(   992322594),
         INT32_C(  1339135030),  INT32_C(   816579021), -INT32_C(  1796784037), -INT32_C(   482380438), -INT32_C(  2003845028),  INT32_C(  1706270209), -INT32_C(  2111009372),  INT32_C(   524682985) },
      {  INT32_C(  1735268250),  INT32_C(  1939347992), -INT32_C(   989298853),  INT32_C(  1353205748),  INT32_C(   148453127),  INT32_C(  2053999829),  INT32_C(    66885914),  INT32_C(   958546847),
         INT32_C(  1939968346),  INT32_C(   115751339), -INT32_C(  1395921224),  INT32_C(  1040020533), -INT32_C(  2126129493),  INT32_C(  2096870242), -INT32_C(   327157940), -INT32_C(  1792695750) },
      {  INT32_C(  1792330876),  INT32_C(   418875182),  INT32_C(   134011108),  INT32_C(   720570282), -INT32_C(  1853596235), -INT32_C(   496598853), -INT32_C(  1204688658), -INT32_C(    34072511),
         INT32_C(  1013974380),  INT32_C(   910833766),  INT32_C(   692331391), -INT32_C(   548194518),  INT32_C(   164239095),  INT32_C(  1177737048),  INT32_C(   603874239), -INT32_C(   681332745) } },
    { { -INT32_C(   569850317), -INT32_C(  1209667841),  INT32_C(   308588765), -INT32_C(   800104156), -INT32_C(  1722706633), -INT32_C(  1810478008),  INT32_C(  2122356035),  INT32_C(  1779672631),
         INT32_C(  1816730476), -INT32_C(   433902071),  INT32_C(    66684894),  INT32_C(   533940456),  INT32_C(   632890589), -INT32_C(  1279602832), -INT32_C(  1674495388),  INT32_C(  1292256480) },
      UINT16_C(20319),
      { -INT32_C(   595760711),  INT32_C(  1214536783),  INT32_C(   848383071),  INT32_C(   626421356),  INT32_C(  1324664468), -INT32_C(  1417127815), -INT32_C(    68195852),  INT32_C(  1867141046),
         INT32_C(   122407096), -INT32_C(  2091864284),  INT32_C(  1756750332), -INT32_C(   477295793),  INT32_C(  1244758481), -INT32_C(   822756647), -INT32_C(   641080029), -INT32_C(   330820812) },
      { -INT32_C(      814117),  INT32_C(  1065501507),  INT32_C(  1957115684),  INT32_C(   324482114), -INT32_C(  1906406987),  INT32_C(  1717326659),  INT32_C(  1816077624),  INT32_C(   341346105),
         INT32_C(  1561611290), -INT32_C(  1264806257),  INT32_C(   271074254),  INT32_C(   740589431),  INT32_C(  1270579720),  INT32_C(   246486997),  INT32_C(  1987768381), -INT32_C(  1836395657) },
      {  INT32_C(   596572770),  INT32_C(  2011569932),  INT32_C(  1177975675),  INT32_C(   906058286), -INT32_C(  1062580447), -INT32_C(  1810478008), -INT32_C(  1747956532),  INT32_C(  1779672631),
         INT32_C(  1516209314),  INT32_C(   936130987),  INT32_C(  2023596594), -INT32_C(   810978248),  INT32_C(   632890589), -INT32_C(  1279602832), -INT32_C(  1347218146),  INT32_C(  1292256480) } },
    { { -INT32_C(  1360028129),  INT32_C(    56790069),  INT32_C(  1192463055),  INT32_C(   309540618), -INT32_C(  1889653063), -INT32_C(  2086858938),  INT32_C(  2012813056),  INT32_C(   151618538),
         INT32_C(  1471740194),  INT32_C(  1415191173), -INT32_C(  1348768347),  INT32_C(  1589710757), -INT32_C(  2081611971),  INT32_C(   788957743), -INT32_C(  1935212638), -INT32_C(  1516851069) },
      UINT16_C(20137),
      {  INT32_C(  1449668348),  INT32_C(   516099458),  INT32_C(  2116905148), -INT32_C(  1264751929), -INT32_C(   197145362), -INT32_C(  1561009669),  INT32_C(    55736173),  INT32_C(   408025884),
        -INT32_C(  1418806743), -INT32_C(  2067189305),  INT32_C(  1627584154),  INT32_C(  1326817121),  INT32_C(  1732465772), -INT32_C(  1559677131), -INT32_C(   895067218),  INT32_C(  2145581142) },
      {  INT32_C(  2032881842),  INT32_C(   486405250),  INT32_C(  1283325931),  INT32_C(   194744991),  INT32_C(   477290215), -INT32_C(   994083818),  INT32_C(   781149911),  INT32_C(   279802206),
         INT32_C(  1149884610), -INT32_C(  1201633331),  INT32_C(   621075846),  INT32_C(  1446027119), -INT32_C(  1821138051), -INT32_C(   161992161), -INT32_C(   165353576),  INT32_C(   436654680) },
      {  INT32_C(   792952398),  INT32_C(    56790069),  INT32_C(  1192463055), -INT32_C(  1090062248), -INT32_C(  1889653063),  INT32_C(  1716231661),  INT32_C(  2012813056),  INT32_C(   150768194),
         INT32_C(  1471740194),  INT32_C(  1017755146),  INT32_C(  1141254940),  INT32_C(   421855246), -INT32_C(  2081611971),  INT32_C(   788957743),  INT32_C(  1015200822), -INT32_C(  1516851069) } },
    { {  INT32_C(  2002686122), -INT32_C(  1657815529),  INT32_C(   197342107),  INT32_C(  1348596690), -INT32_C(  1243360106), -INT32_C(  1616102649),  INT32_C(  2073481250),  INT32_C(  1301650594),
         INT32_C(  1136981036),  INT32_C(  1323431090), -INT32_C(    94788569),  INT32_C(   776649367), -INT32_C(  1763496561), -INT32_C(  1959424151),  INT32_C(    33999712), -INT32_C(  1806721944) },
      UINT16_C( 5264),
      { -INT32_C(  1190640936), -INT32_C(   379768944),  INT32_C(  1940190249),  INT32_C(   111227682), -INT32_C(    23786807), -INT32_C(  1664420459),  INT32_C(  1194865400), -INT32_C(  1638151994),
        -INT32_C(  1688771830), -INT32_C(  1132153709), -INT32_C(   886036311),  INT32_C(   617730395),  INT32_C(  1881368539),  INT32_C(  1410133340), -INT32_C(   442743777),  INT32_C(   394589964) },
      { -INT32_C(   290268325),  INT32_C(   967456400), -INT32_C(  1174087073), -INT32_C(  2032150869), -INT32_C(  1711865283),  INT32_C(   250479599),  INT32_C(  1425246792), -INT32_C(   596936831),
        -INT32_C(   473293485), -INT32_C(  1289980588), -INT32_C(    76734385),  INT32_C(   864111862),  INT32_C(  1036875854), -INT32_C(  1001669764), -INT32_C(   971489467),  INT32_C(   161711286) },
      {  INT32_C(  2002686122), -INT32_C(  1657815529),  INT32_C(   197342107),  INT32_C(  1348596690),  INT32_C(  1734478068), -INT32_C(  1616102649),  INT32_C(  2073481250),  INT32_C(  1110490951),
         INT32_C(  1136981036),  INT32_C(  1323431090),  INT32_C(   811403494),  INT32_C(   776649367),  INT32_C(  1307451285), -INT32_C(  1959424151),  INT32_C(    33999712), -INT32_C(  1806721944) } },
    { { -INT32_C(   169054815),  INT32_C(   849873122),  INT32_C(   523048489), -INT32_C(  1319915934), -INT32_C(  1544675289),  INT32_C(   543636187),  INT32_C(   803635065), -INT32_C(  1522956029),
        -INT32_C(   644209161),  INT32_C(  1460355886), -INT32_C(  1149880231),  INT32_C(   242010599), -INT32_C(   995009559),  INT32_C(   249829525), -INT32_C(  1690449001),  INT32_C(  1262515796) },
      UINT16_C(55964),
      {  INT32_C(   807258661), -INT32_C(  1754696159), -INT32_C(  1637789646),  INT32_C(   267995742), -INT32_C(   215511537), -INT32_C(   625033572), -INT32_C(  1722805415), -INT32_C(  2089489314),
        -INT32_C(   676032074),  INT32_C(   980360456), -INT32_C(   874983315),  INT32_C(   702206490), -INT32_C(    48430752),  INT32_C(   433576895),  INT32_C(  1303521262), -INT32_C(   892328428) },
      { -INT32_C(  1063091016),  INT32_C(   251269537), -INT32_C(    69610783),  INT32_C(    86356900),  INT32_C(  1946305204),  INT32_C(   210622749),  INT32_C(   341393152),  INT32_C(   517941606),
         INT32_C(  1339982254),  INT32_C(  1935595666),  INT32_C(  1332623274), -INT32_C(  1621847061), -INT32_C(   216836395),  INT32_C(   805281839),  INT32_C(  1178884320),  INT32_C(   795091841) },
      { -INT32_C(   169054815),  INT32_C(   849873122),  INT32_C(  1706590931),  INT32_C(   182254074), -INT32_C(  2027565893),  INT32_C(   543636187),  INT32_C(   803635065), -INT32_C(  1649687240),
        -INT32_C(   644209161),  INT32_C(  1227998618), -INT32_C(  1149880231), -INT32_C(  1232190991),  INT32_C(   235820981),  INT32_C(   249829525),  INT32_C(   200703758), -INT32_C(   441186923) } },
    { {  INT32_C(   914309796), -INT32_C(   995500774), -INT32_C(    15526124), -INT32_C(  2137036885), -INT32_C(   327961923),  INT32_C(   857502291),  INT32_C(  1283023050),  INT32_C(   662429059),
         INT32_C(   962460191), -INT32_C(   318961961), -INT32_C(   924118755),  INT32_C(   910789240), -INT32_C(  1876771779), -INT32_C(   104645073),  INT32_C(   558185630),  INT32_C(   944292121) },
      UINT16_C(42427),
      {  INT32_C(  1856738161),  INT32_C(  1786759551), -INT32_C(   621414255),  INT32_C(  1352086062), -INT32_C(  2054175038),  INT32_C(    96546239), -INT32_C(  1748575665), -INT32_C(  2093121262),
        -INT32_C(  1812862956),  INT32_C(  1140748721), -INT32_C(  1759644823), -INT32_C(   404245467),  INT32_C(   980186746), -INT32_C(   213963356),  INT32_C(   428475655), -INT32_C(  1684224122) },
      {  INT32_C(  1630440880),  INT32_C(  1738812670),  INT32_C(  1157546527), -INT32_C(   248781194),  INT32_C(     2856796), -INT32_C(   873239868), -INT32_C(   169574801), -INT32_C(   175079355),
         INT32_C(   207077134),  INT32_C(   192216043),  INT32_C(   877621949), -INT32_C(  1255834792), -INT32_C(   675983342),  INT32_C(   698526138),  INT32_C(  1813940007),  INT32_C(   358788871) },
      {  INT32_C(   260382401),  INT32_C(   232514945), -INT32_C(    15526124), -INT32_C(  1581460392), -INT32_C(  2052828770), -INT32_C(   835565701),  INT32_C(  1283023050),  INT32_C(  1991049559),
        -INT32_C(  1616488678), -INT32_C(   318961961), -INT32_C(  1554873900),  INT32_C(   910789240), -INT32_C(  1876771779), -INT32_C(   627211234),  INT32_C(   558185630), -INT32_C(  1895929727) } },
    { {  INT32_C(  1495447918),  INT32_C(  1919194804),  INT32_C(  1638315016),  INT32_C(  1092012847), -INT32_C(   719795429), -INT32_C(  1677804684),  INT32_C(  1225268290),  INT32_C(   962554571),
        -INT32_C(   678199005),  INT32_C(   524941079), -INT32_C(   629084245), -INT32_C(   719546694), -INT32_C(   693488542),  INT32_C(   829597935), -INT32_C(  1870955835),  INT32_C(   130734820) },
      UINT16_C(23899),
      {  INT32_C(   676623070),  INT32_C(   303497361), -INT32_C(   156708134),  INT32_C(  1378552487),  INT32_C(  1408899809), -INT32_C(   942817460),  INT32_C(   446804304), -INT32_C(  1770521416),
        -INT32_C(     4273298), -INT32_C(  1508780341),  INT32_C(  1335671464), -INT32_C(  1516124220),  INT32_C(   788045026), -INT32_C(  1409956517),  INT32_C(   784701046),  INT32_C(    12926098) },
      { -INT32_C(   738229496),  INT32_C(    24711513), -INT32_C(  1890511157), -INT32_C(  1086983459), -INT32_C(   353555058),  INT32_C(  1771430643),  INT32_C(   177757048), -INT32_C(  1626710633),
         INT32_C(   980617952), -INT32_C(   432280549), -INT32_C(   529167358),  INT32_C(   228567679), -INT32_C(   872968744), -INT32_C(   399209105),  INT32_C(  2146618600),  INT32_C(   169868585) },
      { -INT32_C(    72617514),  INT32_C(   325980616),  INT32_C(  1638315016), -INT32_C(   316737414), -INT32_C(  1189660817), -INT32_C(  1677804684),  INT32_C(   272230952),  INT32_C(   962554571),
        -INT32_C(   976371314),  INT32_C(   524941079), -INT32_C(  1343670614), -INT32_C(  1472303685), -INT32_C(   451997382),  INT32_C(   829597935),  INT32_C(  1362582174),  INT32_C(   130734820) } },
    { {  INT32_C(   574919175), -INT32_C(  2130149506), -INT32_C(  1956545268),  INT32_C(     9961512), -INT32_C(    36990835),  INT32_C(    98959133), -INT32_C(   175843380), -INT32_C(   604003372),
        -INT32_C(  1258470603), -INT32_C(   818608701), -INT32_C(  1420126589),  INT32_C(   615248534), -INT32_C(  1625196926),  INT32_C(  1101268597), -INT32_C(  1305007651),  INT32_C(    42809036) },
      UINT16_C(35706),
      { -INT32_C(   342868554),  INT32_C(  1719735052),  INT32_C(  1767380926),  INT32_C(  1558174523),  INT32_C(   526538106), -INT32_C(   850902890),  INT32_C(  2130973938), -INT32_C(   871727594),
        -INT32_C(   944268357),  INT32_C(  1831680174), -INT32_C(  1948875440), -INT32_C(   605505952), -INT32_C(  1577432565),  INT32_C(  2087666314),  INT32_C(  1845261142), -INT32_C(  1405548815) },
      {  INT32_C(  1349775777),  INT32_C(  2042470441), -INT32_C(  2046455003),  INT32_C(  1415703881), -INT32_C(  1024042185), -INT32_C(   214014563), -INT32_C(   916440360), -INT32_C(   495609279),
        -INT32_C(  1271732085), -INT32_C(  1389432952), -INT32_C(   869059709),  INT32_C(  1461818400), -INT32_C(  1944512529),  INT32_C(  1417697404), -INT32_C(   753016686),  INT32_C(    95851130) },
      {  INT32_C(   574919175),  INT32_C(   524071717), -INT32_C(  1956545268),  INT32_C(   146683506), -INT32_C(   577499571),  INT32_C(  1047930123), -INT32_C(  1234948566), -INT32_C(   604003372),
         INT32_C(  1938125616), -INT32_C(  1073490138), -INT32_C(  1420126589), -INT32_C(  1932975552), -INT32_C(  1625196926),  INT32_C(  1101268597), -INT32_C(  1305007651), -INT32_C(  1450208117) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i src = simde_mm512_loadu_epi64(test_vec[i].src);
    simde__m512i a = simde_mm512_loadu_epi64(test_vec[i].a);
    simde__m512i b = simde_mm512_loadu_epi64(test_vec[i].b);
    simde__m512i r = simde_mm512_castps_si512(simde_mm512_mask_xor_ps(simde_mm512_castsi512_ps(src), test_vec[i].k, simde_mm512_castsi512_ps(a), simde_mm512_castsi512_ps(b)));
    simde_test_x86_assert_equal_i32x16(r, simde_mm512_loadu_epi64(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m512i src = simde_test_x86_random_i32x16();
    simde__mmask16 k = simde_test_x86_random_mmask16();
    simde__m512i a = simde_test_x86_random_i32x16();
    simde__m512i b = simde_test_x86_random_i32x16();
    simde__m512i r = simde_mm512_castps_si512(simde_mm512_mask_xor_ps(simde_mm512_castsi512_ps(src), k, simde_mm512_castsi512_ps(a), simde_mm512_castsi512_ps(b)));

    simde_test_x86_write_i32x16(2, src, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_mmask16(2, k, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x16(2, a, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x16(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x16(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_maskz_xor_ps (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const simde__mmask16 k;
    const int32_t a[16];
    const int32_t b[16];
    const int32_t r[16];
  } test_vec[] = {
    { UINT16_C(59514),
      { -INT32_C(   388496710), -INT32_C(   484746321),  INT32_C(  1215773479), -INT32_C(  1403033710),  INT32_C(  1929698291),  INT32_C(  1280611887),  INT32_C(   551472746), -INT32_C(  1912055596),
         INT32_C(   175562842),  INT32_C(  1676513595),  INT32_C(  1588290764), -INT32_C(  1089860661),  INT32_C(   372379366),  INT32_C(   258115237),  INT32_C(   690962773), -INT32_C(   189319271) },
      {  INT32_C(  1409166616), -INT32_C(  1967723586),  INT32_C(   468214351),  INT32_C(  1406858093), -INT32_C(  1503065087), -INT32_C(   390738797), -INT32_C(  1508776947),  INT32_C(   882559004),
        -INT32_C(  1282959115), -INT32_C(   750961021),  INT32_C(   233711008),  INT32_C(   442550296),  INT32_C(  1740688084), -INT32_C(  1555073386),  INT32_C(  1984520282),  INT32_C(   497738792) },
      {  INT32_C(           0),  INT32_C(  1772990481),  INT32_C(           0), -INT32_C(     8022785), -INT32_C(   714221582), -INT32_C(  1528735044), -INT32_C(  2033243545),  INT32_C(           0),
         INT32_C(           0),  INT32_C(           0),  INT32_C(           0), -INT32_C(  1519729709),  INT32_C(           0), -INT32_C(  1406275533),  INT32_C(  1600528655), -INT32_C(   383921231) } },
    { UINT16_C(12924),
      {  INT32_C(   225444048), -INT32_C(  1053617965),  INT32_C(  2106149661),  INT32_C(   642211173),  INT32_C(   345824709), -INT32_C(   898238848),  INT32_C(   397319533), -INT32_C(  1958139205),
        -INT32_C(    40257238), -INT32_C(   423703351),  INT32_C(  2103723799),  INT32_C(  1789111205),  INT32_C(   159268745), -INT32_C(  1563167947),  INT32_C(  1287291281), -INT32_C(   690551892) },
      { -INT32_C(  2032897859),  INT32_C(  1416467004),  INT32_C(  2144457178),  INT32_C(    99185788), -INT32_C(   401643597), -INT32_C(   326376869),  INT32_C(   272123236),  INT32_C(    98963272),
        -INT32_C(  1131627904),  INT32_C(   655423821),  INT32_C(  1185341898),  INT32_C(   139235157),  INT32_C(  1374772214), -INT32_C(  1589806019),  INT32_C(   162624961),  INT32_C(    84908165) },
      {  INT32_C(           0),  INT32_C(           0),  INT32_C(    39361223),  INT32_C(   598616345), -INT32_C(    57427338),  INT32_C(   654185691),  INT32_C(   127326217),  INT32_C(           0),
         INT32_C(           0), -INT32_C(  1045547644),  INT32_C(           0),  INT32_C(           0),  INT32_C(  1485792383),  INT32_C(    65965832),  INT32_C(           0),  INT32_C(           0) } },
    { UINT16_C(39762),
      { -INT32_C(   762011711),  INT32_C(  1823694534), -INT32_C(   251983452),  INT32_C(    38531601), -INT32_C(  2122413757), -INT32_C(   587841750),  INT32_C(  1484028745),  INT32_C(  1123272320),
         INT32_C(   739542886), -INT32_C(  1986410523), -INT32_C(   495282992), -INT32_C(   890976890),  INT32_C(  2034983758), -INT32_C(   346734174),  INT32_C(  1027852733), -INT32_C(   176212337) },
      { -INT32_C(  1574792259),  INT32_C(   724351835), -INT32_C(   703748529), -INT32_C(  1163857301), -INT32_C(   147592363), -INT32_C(   370964436), -INT32_C(   534305199),  INT32_C(   450274909),
        -INT32_C(  1782712262),  INT32_C(    46197171), -INT32_C(    86454641),  INT32_C(   364148928), -INT32_C(  1878202525), -INT32_C(  1065750673),  INT32_C(  1939972118), -INT32_C(  2121435321) },
      {  INT32_C(           0),  INT32_C(  1201661341),  INT32_C(           0),  INT32_C(           0),  INT32_C(  1984782870),  INT32_C(           0), -INT32_C(  1202496232),  INT32_C(           0),
        -INT32_C(  1180074148), -INT32_C(  1957089706),  INT32_C(           0), -INT32_C(   548356794), -INT32_C(   381189075),  INT32_C(           0),  INT32_C(           0),  INT32_C(  1962033608) } },
    { UINT16_C(19055),
      { -INT32_C(   701292010), -INT32_C(    39534044),  INT32_C(  1903518909), -INT32_C(  2040931975), -INT32_C(   495597463),  INT32_C(   713264009), -INT32_C(  1935553794),  INT32_C(  1641418827),
         INT32_C(  1446447666), -INT32_C(  1990992692), -INT32_C(  1157904064),  INT32_C(   188765346), -INT32_C(  1511148260),  INT32_C(  1070559297), -INT32_C(  2050265030), -INT32_C(  1293507968) },
      {  INT32_C(  2013863596),  INT32_C(   973233402), -INT32_C(   940245723),  INT32_C(  1842492497),  INT32_C(   722714858),  INT32_C(  1802166832), -INT32_C(   739232173), -INT32_C(  2071537704),
        -INT32_C(   268660747),  INT32_C(   287964907),  INT32_C(  1289232123),  INT32_C(  1018866514), -INT32_C(  1687696021),  INT32_C(    34001583), -INT32_C(   522782968), -INT32_C(  1016832818) },
      { -INT32_C(  1371915078), -INT32_C(   945381666), -INT32_C(  1233020520), -INT32_C(   343151320),  INT32_C(           0),  INT32_C(  1105815993),  INT32_C(  1599208621),  INT32_C(           0),
         INT32_C(           0), -INT32_C(  1736826329),  INT32_C(           0),  INT32_C(   939196400),  INT32_C(           0),  INT32_C(           0),  INT32_C(  1696433970),  INT32_C(           0) } },
    { UINT16_C(24811),
      { -INT32_C(   614476110), -INT32_C(  1057400089),  INT32_C(  1634421927), -INT32_C(   265365880),  INT32_C(  2026036593), -INT32_C(  1234187552),  INT32_C(   236076458), -INT32_C(  1301349120),
        -INT32_C(  1148334637), -INT32_C(   813988056),  INT32_C(  1529931475),  INT32_C(   759914172),  INT32_C(   463801659), -INT32_C(  2117004073),  INT32_C(  1368384337), -INT32_C(  1274806560) },
      { -INT32_C(   210791989), -INT32_C(   339547624), -INT32_C(  1941507376), -INT32_C(  1933995440),  INT32_C(  2007457695), -INT32_C(   973571725),  INT32_C(  1024886876),  INT32_C(  1374755462),
        -INT32_C(  1018863445),  INT32_C(   447612746),  INT32_C(  1252455674),  INT32_C(   651583366),  INT32_C(   849182398),  INT32_C(  1408734711), -INT32_C(  1550840547), -INT32_C(   755662553) },
      {  INT32_C(   674252665),  INT32_C(   725332223),  INT32_C(           0),  INT32_C(  2090288344),  INT32_C(           0),  INT32_C(  1939321747),  INT32_C(   855946742), -INT32_C(   476059770),
         INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0), -INT32_C(   769228512), -INT32_C(   232788404),  INT32_C(           0) } },
    { UINT16_C(15073),
      {  INT32_C(  1128344469), -INT32_C(   348636347),  INT32_C(  1548467846), -INT32_C(  2116417052),  INT32_C(   840421691), -INT32_C(  1254083547), -INT32_C(   852006952), -INT32_C(   821618630),
        -INT32_C(  1995225020),  INT32_C(   175459204), -INT32_C(   295256055),  INT32_C(    91177417),  INT32_C(   926385682), -INT32_C(  1813153861),  INT32_C(   425731295), -INT32_C(  2132252868) },
      {  INT32_C(   856357807),  INT32_C(  1346207558),  INT32_C(   138323007), -INT32_C(   150098459), -INT32_C(   282114764), -INT32_C(  1685971780),  INT32_C(  2092229184), -INT32_C(   117596855),
        -INT32_C(   550762600), -INT32_C(   986748538), -INT32_C(   221418227),  INT32_C(  1324014362), -INT32_C(   599975648),  INT32_C(  1970847541), -INT32_C(   353293151),  INT32_C(  1659104969) },
      {  INT32_C(  1884016698),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(   784476313), -INT32_C(  1316780648),  INT32_C(   939165043),
         INT32_C(           0), -INT32_C(   816176638),  INT32_C(           0),  INT32_C(  1267047123), -INT32_C(   351625422), -INT32_C(   426391410),  INT32_C(           0),  INT32_C(           0) } },
    { UINT16_C( 4085),
      {  INT32_C(  1886944065),  INT32_C(   249398848),  INT32_C(  1659500408), -INT32_C(  2089088698),  INT32_C(  1564717285),  INT32_C(   394978341), -INT32_C(  1308273713), -INT32_C(   155059275),
        -INT32_C(  1218037386),  INT32_C(   952452031), -INT32_C(  2120569285), -INT32_C(  1677453641), -INT32_C(   319141946), -INT32_C(    83655636), -INT32_C(  1918039849),  INT32_C(  2055433731) },
      {  INT32_C(  1731324583),  INT32_C(  1755313709),  INT32_C(  1542011300),  INT32_C(   385347151),  INT32_C(  1627582773),  INT32_C(  1297876341),  INT32_C(   299501837),  INT32_C(   529227127),
         INT32_C(  1971764295),  INT32_C(  1457333682), -INT32_C(  1364080802), -INT32_C(   356210507),  INT32_C(   256689818), -INT32_C(   665016117),  INT32_C(   703149746), -INT32_C(   616008556) },
      {  INT32_C(   390697446),  INT32_C(           0),  INT32_C(   956354268),  INT32_C(           0),  INT32_C(  1010909648),  INT32_C(  1524031824), -INT32_C(  1545643326), -INT32_C(   380983614),
        -INT32_C(  1025473231),  INT32_C(  1847092749),  INT32_C(   791374181),  INT32_C(  1992343042),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0) } },
    { UINT16_C(52784),
      {  INT32_C(   787735120), -INT32_C(   369798600), -INT32_C(   997021184), -INT32_C(   527815532), -INT32_C(  1735830213),  INT32_C(   399391533), -INT32_C(  1383374236), -INT32_C(  1904493506),
        -INT32_C(   675516514), -INT32_C(  1044336191), -INT32_C(   259698085), -INT32_C(  1160769666), -INT32_C(  1856874140), -INT32_C(   106422124), -INT32_C(  1046072189), -INT32_C(  1907416592) },
      {  INT32_C(  1365576848),  INT32_C(   420619965), -INT32_C(   150366344),  INT32_C(   179427750), -INT32_C(   946076877), -INT32_C(  1497349085),  INT32_C(  1751606904),  INT32_C(   418887560),
        -INT32_C(  2140513085), -INT32_C(    73827198), -INT32_C(  1158503916), -INT32_C(  1346002052), -INT32_C(   914988634),  INT32_C(   510670757),  INT32_C(   629594013),  INT32_C(  1363049870) },
      {  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(  1595233800), -INT32_C(  1324450034),  INT32_C(           0),  INT32_C(           0),
         INT32_C(           0),  INT32_C(   978963779),  INT32_C(  1249374287),  INT32_C(   353741826),  INT32_C(           0),  INT32_C(           0), -INT32_C(   467606754), -INT32_C(   546218882) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i a = simde_mm512_loadu_epi64(test_vec[i].a);
    simde__m512i b = simde_mm512_loadu_epi64(test_vec[i].b);
    simde__m512i r = simde_mm512_castps_si512(simde_mm512_maskz_xor_ps(test_vec[i].k, simde_mm512_castsi512_ps(a), simde_mm512_castsi512_ps(b)));
    simde_test_x86_assert_equal_i32x16(r, simde_mm512_loadu_epi64(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__mmask16 k = simde_test_x86_random_mmask16();
    simde__m512i a = simde_test_x86_random_i32x16();
    simde__m512i b = simde_test_x86_random_i32x16();
    simde__m512i r = simde_mm512_castps_si512(simde_mm512_maskz_xor_ps(k, simde_mm512_castsi512_ps(a), simde_mm512_castsi512_ps(b)));

    simde_test_x86_write_mmask16(2, k, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_i32x16(2, a, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x16(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i32x16(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_xor_pd (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int64_t a[8];
    const int64_t b[8];
    const int64_t r[8];
  } test_vec[] = {
    { { -INT64_C( 8917272025905183984), -INT64_C( 8866677502414823733),  INT64_C( 1464795012987523672),  INT64_C( 9076492970404562796),
        -INT64_C(  471420776636169871),  INT64_C( 3961263463326435322), -INT64_C( 4926361178749313089),  INT64_C( 6187424904041771752) },
      { -INT64_C( 6187074978812186357),  INT64_C( 9129308580762170105),  INT64_C( 6737626754639454484),  INT64_C( 8553348640616316182),
        -INT64_C( 7747167415395374764),  INT64_C( 2270104474263871152),  INT64_C( 8997904634902970919), -INT64_C( 1284389140319940514) },
      {  INT64_C( 3322640854544675355), -INT64_C(  413510917362488270),  INT64_C( 5319635821359362892),  INT64_C(  812256522989615738),
         INT64_C( 7856992465972914213),  INT64_C( 2988204169892698442), -INT64_C( 4072109430317720168), -INT64_C( 4903636269142318410) } },
    { { -INT64_C( 6505531527298525732),  INT64_C( 9073156322736741803),  INT64_C( 3499178133975168607),  INT64_C( 8184285759661098913),
         INT64_C( 3319482116807104653),  INT64_C( 6307874133959206792), -INT64_C( 1590393193835026219),  INT64_C( 3334991353612573190) },
      { -INT64_C(  189158219624270114),  INT64_C( 4609744475273820848),  INT64_C( 3508270164975553796), -INT64_C( 7960726155283276816),
         INT64_C( 2082343515212321040),  INT64_C( 4039683120892358364), -INT64_C( 7716078895957568331),  INT64_C( 8874843879011272792) },
      {  INT64_C( 6406454113495114498),  INT64_C( 4761230285051520795),  INT64_C(    9133478509509467), -INT64_C( 2300849903224578991),
         INT64_C( 3671791598235879837),  INT64_C( 8036045162057386324),  INT64_C( 9009225591048387680),  INT64_C( 6152343202689438302) } },
    { {  INT64_C( 1660001668875280251), -INT64_C( 6688666841194484293),  INT64_C(  866481487179763680), -INT64_C( 6014321076695304337),
        -INT64_C( 4943803188177761355),  INT64_C( 8602557963703392155),  INT64_C( 2348766465129802213), -INT64_C( 4226418528419391895) },
      { -INT64_C(  276208556333754517), -INT64_C( 8793904538853466362), -INT64_C( 1578788849987643064), -INT64_C( 1646247322702001477),
         INT64_C( 2188768063495513738),  INT64_C( 7993857117939491420), -INT64_C( 8683858962037236194),  INT64_C( 8744573975962379253) },
      { -INT64_C( 1503296891938387952),  INT64_C( 2799218735084909245), -INT64_C( 1868607137191635288),  INT64_C( 5021386112833600980),
        -INT64_C( 6556108815733478081),  INT64_C( 1841284304668030407), -INT64_C( 6348735240143046661), -INT64_C( 4899273116385921636) } },
    { { -INT64_C( 7645977387302788113), -INT64_C( 3352234231512672247),  INT64_C( 7099980801611090558),  INT64_C( 2943395267179314445),
        -INT64_C( 4607844101183449547), -INT64_C( 8977088685291950508),  INT64_C(  143793199971752471),  INT64_C( 1279544180712774249) },
      {  INT64_C( 5042683055231339073),  INT64_C(  106425879644314133), -INT64_C( 7957364987157442167),  INT64_C( 2375707624005304798),
         INT64_C( 1229740195508336126), -INT64_C( 4558434066425528247), -INT64_C( 6013628711494954681),  INT64_C(  863595646972066585) },
      { -INT64_C( 3449971071047439954), -INT64_C( 3458657327579413988), -INT64_C(  929447554171085833),  INT64_C(  585818806074473683),
        -INT64_C( 3378460560296254517),  INT64_C( 4888597167206592029), -INT64_C( 5947680971553895600),  INT64_C( 1890879682201149808) } },
    { {  INT64_C(   19342552227360650),  INT64_C( 9015339881571254999),  INT64_C( 2111175543395945328),  INT64_C( 1063893217915658645),
         INT64_C( 5113671542448273537),  INT64_C(  294076048327577819),  INT64_C( 4136299984689877214),  INT64_C( 6998346057415234657) },
      { -INT64_C( 5822897953418392471), -INT64_C( 2652605089542910307), -INT64_C( 6710896084442025600),  INT64_C(  482490459503952710),
        -INT64_C( 4249188140389718191),  INT64_C( 3686292135995040188), -INT64_C( 7820050545947625371), -INT64_C( 4155849880699296371) },
      { -INT64_C( 5803909723175885853), -INT64_C( 6472545252534039478), -INT64_C( 4642510373644240144),  INT64_C(  608429854075970259),
        -INT64_C( 8939450874709434416),  INT64_C( 3980218559741096295), -INT64_C( 6188350928172226373), -INT64_C( 6391617066844278292) } },
    { {  INT64_C( 6004850654787511453),  INT64_C( 5940721718117239162),  INT64_C( 8299177194198841098),  INT64_C( 8739678760146743174),
        -INT64_C(   33468914264828954),  INT64_C( 4810661481483717294), -INT64_C( 6371855048832433144),  INT64_C( 2475934475524100073) },
      {  INT64_C( 6400425042904156857),  INT64_C( 7634775463157369383), -INT64_C( 8012522724089233327),  INT64_C( 3036153849740553193),
        -INT64_C( 1737102447742047799), -INT64_C( 2861167923869488012),  INT64_C(  265573619744849241),  INT64_C( 8622106189528170193) },
      {  INT64_C(  830741991547514404),  INT64_C( 4288980697190663005), -INT64_C( 2026245280167086245),  INT64_C( 6010905068804162671),
         INT64_C( 1760215543021156911), -INT64_C( 7311046646325205286), -INT64_C( 6612095300032300719),  INT64_C( 6195711379304100152) } },
    { { -INT64_C(   64316270072716246),  INT64_C( 1935443446427172380), -INT64_C( 1091493333936354380), -INT64_C( 6895415197380722231),
        -INT64_C( 1805936298755591266),  INT64_C( 8997027290326566875), -INT64_C( 7447454384865349272),  INT64_C( 6845659238006585079) },
      { -INT64_C( 3918924663269530680), -INT64_C( 5736519009328467646),  INT64_C( 3896138832546275085),  INT64_C(  114649980956301072),
        -INT64_C( 3878013855955832913), -INT64_C( 8817041431659451413), -INT64_C( 3144109803129131596), -INT64_C( 3898751961541556299) },
      {  INT64_C( 3929019560870856674), -INT64_C( 6142951854191587490), -INT64_C( 4121953400583433031), -INT64_C( 6784178172063836455),
         INT64_C( 3233162805184420913), -INT64_C(  470539099348898768),  INT64_C( 5546380282716151004), -INT64_C( 7573801740836425406) } },
    { {  INT64_C( 4005776973137074199),  INT64_C( 7761109657262433250), -INT64_C( 4087271545684412689),  INT64_C( 8176212645445862943),
         INT64_C( 1663312718869912950), -INT64_C( 5987727983636667819), -INT64_C( 2064163572997672891),  INT64_C( 4576071772732176903) },
      {  INT64_C( 3480407709618747496),  INT64_C( 2300011067054476999),  INT64_C( 2573836082898378693), -INT64_C( 8237295633617103203),
         INT64_C( 8697230954808831974), -INT64_C( 7715133602037231102),  INT64_C( 1008851380274702642), -INT64_C( 7795902896402923754) },
      {  INT64_C(  566192077266575487),  INT64_C( 8385218711665306917), -INT64_C( 1945833864310642390), -INT64_C(  227289569249472382),
         INT64_C( 8045543724174291600),  INT64_C( 4037767070791741527), -INT64_C( 1343560506274506377), -INT64_C( 6030853617239309039) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i a = simde_mm512_loadu_epi64(test_vec[i].a);
    simde__m512i b = simde_mm512_loadu_epi64(test_vec[i].b);
    simde__m512i r = simde_mm512_castpd_si512(simde_mm512_xor_pd(simde_mm512_castsi512_pd(a), simde_mm512_castsi512_pd(b)));
    simde_test_x86_assert_equal_i64x8(r, simde_mm512_loadu_epi64(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm512_mask_xor_pd (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const int64_t src[8];
    const simde__mmask8 k;
    const int64_t a[8];
    const int64_t b[8];
    const int64_t r[8];
  } test_vec[] = {
    { {  INT64_C( 6437586905583906514), -INT64_C( 5153936525484381450),  INT64_C( 6943270933310206069),  INT64_C( 7524211599809177408),
         INT64_C( 5824434538339005260),  INT64_C( 7780027683751815990), -INT64_C( 1394359224491689947), -INT64_C( 3449163012513913563) },
      UINT8_C(143),
      { -INT64_C( 8495133462940267688), -INT64_C( 8336185638516663307),  INT64_C( 7003137260866016717),  INT64_C( 7314044866347264725),
        -INT64_C( 7518365256422627072), -INT64_C( 2116391571925423679), -INT64_C( 3295476051057190490), -INT64_C( 2415543629020714745) },
      { -INT64_C( 1376229935454239572),  INT64_C( 6901383497781455056), -INT64_C( 2860763433324596356),  INT64_C(   30230331557417135),
        -INT64_C( 2431603988126275541),  INT64_C( 7131038683452621789),  INT64_C( 7918781416053946429),  INT64_C( 7173219600052179513) },
      {  INT64_C( 7421262813812002804), -INT64_C( 3203949473688325339), -INT64_C( 5081007143180949839),  INT64_C( 7344196020119140986),
         INT64_C( 5824434538339005260),  INT64_C( 7780027683751815990), -INT64_C( 1394359224491689947), -INT64_C( 4758573184432080066) } },
    { { -INT64_C( 1244407392406742067),  INT64_C( 4865711713716370749), -INT64_C( 4260908224916397389),  INT64_C(  766981091718453256),
        -INT64_C( 7658195962633560647),  INT64_C( 7738943069788650780), -INT64_C(  652534938857588831), -INT64_C( 2748924064059407726) },
      UINT8_C( 57),
      { -INT64_C( 7387267490389056577), -INT64_C( 4845665373647692064),  INT64_C( 2442816728953712025),  INT64_C( 5687231108945655512),
        -INT64_C( 8740879232912228419),  INT64_C( 6523012989000680390), -INT64_C( 6245484632013393837),  INT64_C( 7865618868345869642) },
      { -INT64_C( 4326489205870504503),  INT64_C( 6005009147614894651), -INT64_C( 2035670358823730107),  INT64_C( 3497860287354969197),
        -INT64_C( 8972132218061254469),  INT64_C( 7170527882349840943),  INT64_C( 5572422650585221081), -INT64_C( 5554076370592984492) },
      {  INT64_C( 6525187708755080822),  INT64_C( 4865711713716370749), -INT64_C( 4260908224916397389),  INT64_C( 9108530150424488629),
         INT64_C(  418480647996337926),  INT64_C( 4108603231454235113), -INT64_C(  652534938857588831), -INT64_C( 2748924064059407726) } },
    { {  INT64_C( 5751350845764241924), -INT64_C(  111744307068776463),  INT64_C( 8734385293935832072), -INT64_C( 7385587185623981865),
        -INT64_C( 4316499222092911446), -INT64_C( 6905909834716768507),  INT64_C( 3950430239385557299), -INT64_C( 3418646465755150753) },
      UINT8_C(115),
      {  INT64_C( 5273446548515559796),  INT64_C( 2043087781258943199), -INT64_C( 8419001745006677494),  INT64_C( 2284030533521448375),
        -INT64_C( 2011568597036828390), -INT64_C( 9047058233431492909),  INT64_C( 6279836063075462679),  INT64_C( 8312880031040198546) },
      { -INT64_C(  925114651358812351), -INT64_C( 5816755603916639157), -INT64_C( 7376490438547762519), -INT64_C( 4605997696600819895),
        -INT64_C(  797366201897102116), -INT64_C( 7085289264883681181), -INT64_C( 5750947845152246128),  INT64_C( 1872765433263818942) },
      { -INT64_C( 5042238749766304203), -INT64_C( 5540489550081003884),  INT64_C( 8734385293935832072), -INT64_C( 7385587185623981865),
         INT64_C( 1223385550624173510),  INT64_C( 2295035409827286704), -INT64_C( 1794980464735676281), -INT64_C( 3418646465755150753) } },
    { { -INT64_C( 7685388149650553926), -INT64_C( 7004158388354412484), -INT64_C(  483744182612675791), -INT64_C( 4844971588070237331),
         INT64_C( 1801009789046675511), -INT64_C(  345628411063398776),  INT64_C( 4394350837817960903), -INT64_C( 7189809452249894535) },
      UINT8_C(130),
      { -INT64_C( 7901434687957039784), -INT64_C( 8387014073944052033), -INT64_C( 3872086771146056212),  INT64_C( 5740063086953348988),
         INT64_C( 4496694143872862161),  INT64_C(  437603542766895973),  INT64_C( 9056197899055033163),  INT64_C( 8870204133704195539) },
      {  INT64_C( 1445857446161561683),  INT64_C( 2061432370105099130),  INT64_C( 5951135818520226584), -INT64_C( 2832705319378077300),
         INT64_C( 3544538202363265133), -INT64_C( 7941146346981504035),  INT64_C(  476890604075928524),  INT64_C( 6834191479376427898) },
      { -INT64_C( 7685388149650553926), -INT64_C( 7565771451311600187), -INT64_C(  483744182612675791), -INT64_C( 4844971588070237331),
         INT64_C( 1801009789046675511), -INT64_C(  345628411063398776),  INT64_C( 4394350837817960903),  INT64_C( 2724298826198836905) } },
    { {  INT64_C( 1861582297933447660), -INT64_C( 3926976160503129226), -INT64_C( 1810812724388222370), -INT64_C(  955399710856373650),
         INT64_C( 6454903885903836990),  INT64_C( 5606358040040725511),  INT64_C(  307375714144281330),  INT64_C( 2672641076692723998) },
      UINT8_C( 87),
      { -INT64_C( 8133057604378517587), -INT64_C( 3128369023641836069),  INT64_C( 6995617683314701298), -INT64_C(  123729894868305456),
        -INT64_C( 1094594520523767284),  INT64_C(  802529247776427357), -INT64_C( 3768360624300323556),  INT64_C( 8403439177621568592) },
      {  INT64_C( 5515365758784550533),  INT64_C( 7130815896315764176),  INT64_C( 6850654592749139788),  INT64_C( 6215987634154556774),
         INT64_C( 5373211595502647125),  INT64_C( 4125625930493472153), -INT64_C( 4138928590701922783), -INT64_C( 8968962439989850273) },
      { -INT64_C( 4347362476010100440), -INT64_C( 5305113142871574517),  INT64_C( 4469549099567023294), -INT64_C(  955399710856373650),
        -INT64_C( 5017498788745022119),  INT64_C( 5606358040040725511),  INT64_C(  953502644259945277),  INT64_C( 2672641076692723998) } },
    { { -INT64_C( 8317205185373374257),  INT64_C( 1417416665039882374),  INT64_C( 2146759881640848291),  INT64_C(  437926923877572449),
        -INT64_C( 4986361973360829620), -INT64_C( 8669190806303917910),  INT64_C( 2872827924448015107),  INT64_C( 5544434721573252020) },
      UINT8_C(165),
      { -INT64_C( 7940528556938261440),  INT64_C(  980655630240278975),  INT64_C( 7318906000190430344), -INT64_C( 5971576922601849679),
         INT64_C( 5487808273607293246),  INT64_C( 2534539895022788102), -INT64_C( 6090395042801713190), -INT64_C( 3447708299583561971) },
      { -INT64_C( 7414736347008076223),  INT64_C( 6263633501593599278), -INT64_C( 8463701320435831580),  INT64_C( 4117745779878441412),
         INT64_C( 5426028437997849889), -INT64_C( 3463548576539081187),  INT64_C(  212575448128707052),  INT64_C( 4835596213675358185) },
      {  INT64_C(  636137617506597377),  INT64_C( 1417416665039882374), -INT64_C( 1217345983919574932),  INT64_C(  437926923877572449),
        -INT64_C( 4986361973360829620), -INT64_C( 1386125238534170597),  INT64_C( 2872827924448015107), -INT64_C( 7837324708450129692) } },
    { {  INT64_C( 6949693325426827786),  INT64_C( 8216510233329632843), -INT64_C( 3061938756482517951),  INT64_C( 6929581147847842245),
        -INT64_C( 5505939268427111693), -INT64_C( 5154386854511210625), -INT64_C( 1916922516866575708), -INT64_C( 2285305309217154158) },
      UINT8_C( 25),
      {  INT64_C( 4271440688287127247),  INT64_C( 1563430506160344094), -INT64_C( 4328211410770902365), -INT64_C( 1155577783541245978),
        -INT64_C( 3850578045069396057),  INT64_C( 5703118074251067724),  INT64_C( 6573174584934947533),  INT64_C( 2270428257532307570) },
      { -INT64_C( 8855481797091050843),  INT64_C( 5494315124780299075), -INT64_C(  439671032366634220), -INT64_C( 1046790816815964703),
         INT64_C( 3532614236670979143),  INT64_C( 7645811813566749521), -INT64_C( 5274602650662085556), -INT64_C( 6038331607380633610) },
      { -INT64_C( 4729845492485051286),  INT64_C( 8216510233329632843), -INT64_C( 3061938756482517951),  INT64_C( 2202151832208001543),
        -INT64_C(  321547477604197408), -INT64_C( 5154386854511210625), -INT64_C( 1916922516866575708), -INT64_C( 2285305309217154158) } },
    { {  INT64_C( 2635789764425250421),  INT64_C( 7173969323519948495),  INT64_C( 5870444707384296172), -INT64_C( 8760690870588307366),
        -INT64_C( 7190984385298327236),  INT64_C( 2643138234015623219),  INT64_C( 4719594182013215955),  INT64_C( 1548230951492915941) },
      UINT8_C(  4),
      { -INT64_C( 9072943066938006171), -INT64_C(  913142141260504785),  INT64_C( 1298755229924884931), -INT64_C( 2338494684052390629),
        -INT64_C(  432107195710939868),  INT64_C( 6574355609988807225),  INT64_C( 5575600590773032526),  INT64_C( 8059297408314788736) },
      { -INT64_C( 6015985071739266371),  INT64_C( 1844841742087880103),  INT64_C( 1929731531138864967), -INT64_C( 5130465311139325165),
         INT64_C( 5595374708968176006),  INT64_C( 1415018467375750080), -INT64_C( 1805287231774192834),  INT64_C( 8692810336975390191) },
      {  INT64_C( 2635789764425250421),  INT64_C( 7173969323519948495),  INT64_C(  631018082675824260), -INT64_C( 8760690870588307366),
        -INT64_C( 7190984385298327236),  INT64_C( 2643138234015623219),  INT64_C( 4719594182013215955),  INT64_C( 1548230951492915941) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i src = simde_mm512_loadu_epi64(test_vec[i].src);
    simde__m512i a = simde_mm512_loadu_epi64(test_vec[i].a);
    simde__m512i b = simde_mm512_loadu_epi64(test_vec[i].b);
    simde__m512i r = simde_mm512_castpd_si512(simde_mm512_mask_xor_pd(simde_mm512_castsi512_pd(src), test_vec[i].k, simde_mm512_castsi512_pd(a), simde_mm512_castsi512_pd(b)));
    simde_test_x86_assert_equal_i64x8(r, simde_mm512_loadu_epi64(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__m512i src = simde_test_x86_random_i64x8();
    simde__mmask8 k = simde_test_x86_random_mmask8();
    simde__m512i a = simde_test_x86_random_i64x8();
    simde__m512i b = simde_test_x86_random_i64x8();
    simde__m512i r = simde_mm512_castpd_si512(simde_mm512_mask_xor_pd(simde_mm512_castsi512_pd(src), k, simde_mm512_castsi512_pd(a), simde_mm512_castsi512_pd(b)));

    simde_test_x86_write_i64x8(2, src, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_mmask8(2, k, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i64x8(2, a, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i64x8(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i64x8(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_maskz_xor_pd (SIMDE_MUNIT_TEST_ARGS) {
#if 1
  static const struct {
    const simde__mmask8 k;
    const int64_t a[8];
    const int64_t b[8];
    const int64_t r[8];
  } test_vec[] = {
    { UINT8_C(178),
      {  INT64_C( 4535000673791092910), -INT64_C( 3010281343767629417), -INT64_C( 8904038272174085999),  INT64_C( 2275293563450029107),
         INT64_C( 3707561576362413344),  INT64_C( 1693514566972551576), -INT64_C( 4027204785809648325), -INT64_C( 2377269758930565262) },
      {  INT64_C( 8798212751342500426),  INT64_C( 4635791733872161720),  INT64_C( 6203697368372176313),  INT64_C( 5321913355302715709),
         INT64_C( 3355262493918293813),  INT64_C( 1297812233894278332),  INT64_C( 1035961853024525095),  INT64_C(  685644267943270714) },
      {  INT64_C(                   0), -INT64_C( 7607440601921174993),  INT64_C(                   0),  INT64_C(                   0),
         INT64_C( 2153757367459176981),  INT64_C(  396971936153175332),  INT64_C(                   0), -INT64_C( 2989870719856222648) } },
    { UINT8_C(212),
      {  INT64_C( 7330022570064486579),  INT64_C( 6470776002694773732), -INT64_C( 3322187338369804494),  INT64_C( 6195738823362944834),
        -INT64_C( 6781283329911994929), -INT64_C( 6531323974896934267),  INT64_C(   68176035082962666),  INT64_C( 1606415384555526228) },
      {  INT64_C( 3298177160410668391),  INT64_C(  591251513971575580),  INT64_C( 1067384700368749586), -INT64_C( 9218498889734817868),
         INT64_C( 4450234974760904685), -INT64_C( 8026820267700522380), -INT64_C( 7616096171536104058), -INT64_C( 7944169816828259545) },
      {  INT64_C(                   0),  INT64_C(                   0), -INT64_C( 2362942239413349600),  INT64_C(                   0),
        -INT64_C( 7194953829069757918),  INT64_C(                   0), -INT64_C( 7585181778073074836), -INT64_C( 8679701041803572365) } },
    { UINT8_C( 74),
      { -INT64_C( 2719705882611210652),  INT64_C( 3890521222854268022), -INT64_C( 2332954163621184184),  INT64_C(  229144407503527744),
        -INT64_C( 5116632051351512773), -INT64_C( 8607455037759662106), -INT64_C( 7937981019297111189), -INT64_C( 6187203240389158273) },
      { -INT64_C( 1497502927574600900),  INT64_C(   66373760537749284),  INT64_C(  816503436927822961), -INT64_C( 6250629679042344665),
         INT64_C(   13616198539808576), -INT64_C( 5594509499254410553),  INT64_C( 5247759133439251259), -INT64_C( 8455473893407261293) },
      {  INT64_C(                   0),  INT64_C( 3825287698824475474),  INT64_C(                   0), -INT64_C( 6165611562172606873),
         INT64_C(                   0),  INT64_C(                   0), -INT64_C( 2808722857730379696),  INT64_C(                   0) } },
    { UINT8_C( 47),
      {  INT64_C( 8284103234516930512),  INT64_C(  214877212740113264), -INT64_C( 7838960910472133926), -INT64_C( 5794047091137497527),
         INT64_C( 2281872209261601591), -INT64_C( 7144158427984536279), -INT64_C( 5709308242018960228),  INT64_C( 4699981245450855084) },
      {  INT64_C( 1666652842254473299),  INT64_C( 8493447882235445120),  INT64_C( 4601952674112740806), -INT64_C(  551686946749221619),
         INT64_C( 8368281792577982748), -INT64_C( 6815752888624213733), -INT64_C( 8681344165041270862),  INT64_C( 7944058725472292768) },
      {  INT64_C( 7338105342941910915),  INT64_C( 8585457891708546288), -INT64_C( 5986690151986438372),  INT64_C( 6327394179636249412),
         INT64_C(                   0),  INT64_C( 4445982027659365426),  INT64_C(                   0),  INT64_C(                   0) } },
    { UINT8_C(200),
      { -INT64_C( 6021539633559668423), -INT64_C( 2581080677709028376),  INT64_C( 5206969282928735478), -INT64_C(   55158347487606660),
         INT64_C( 1353827748649695618),  INT64_C( 8128924183901983015), -INT64_C( 9191192612159019471), -INT64_C( 7448619434912772645) },
      { -INT64_C( 2276993227150723799), -INT64_C( 1965359776232376881), -INT64_C(  482587615769474359), -INT64_C( 7332918534963773042),
         INT64_C( 4596860684338131250),  INT64_C(  855474210167365449), -INT64_C(  206279998555230709),  INT64_C( 3347263021807418983) },
      {  INT64_C(                   0),  INT64_C(                   0),  INT64_C(                   0),  INT64_C( 7277874962021599730),
         INT64_C(                   0),  INT64_C(                   0),  INT64_C( 9030128929851689018), -INT64_C( 5272890609481960516) } },
    { UINT8_C(206),
      { -INT64_C( 8863509690588522143), -INT64_C( 4084252655939553020),  INT64_C( 6644273578117593769), -INT64_C( 2125833707023036716),
         INT64_C( 4397929110397500575),  INT64_C( 5201328400761081486),  INT64_C( 7730512975766119502), -INT64_C( 4270086284785180838) },
      {  INT64_C( 8632926499810858072),  INT64_C( 8360102739033674014), -INT64_C( 4506510691621921934), -INT64_C( 8849475216152424121),
        -INT64_C( 1869305245149529939),  INT64_C( 7840671269508082998),  INT64_C( 2300754294035853252), -INT64_C( 8057294997331146372) },
      {  INT64_C(                   0), -INT64_C( 5524547863824455654), -INT64_C( 7115515909856927269),  INT64_C( 7444411344080314259),
         INT64_C(                   0),  INT64_C(                   0),  INT64_C( 8405308176221621130),  INT64_C( 6094258454041925158) } },
    { UINT8_C( 52),
      {  INT64_C( 2476948137554290500), -INT64_C( 5047539530763289977), -INT64_C( 7681258544912326657), -INT64_C( 8994804189721259973),
        -INT64_C( 7816266480404911824),  INT64_C( 3101210119854334917), -INT64_C( 8829451883390854443),  INT64_C( 2904164220738072460) },
      { -INT64_C( 3858999919556822574), -INT64_C( 4535086428222573128),  INT64_C( 3282026407837967804), -INT64_C( 8923243080841688823),
        -INT64_C( 8931248512236865729),  INT64_C(  483706269034096393),  INT64_C( 8893932455578065054), -INT64_C( 1596624292404238943) },
      {  INT64_C(                   0),  INT64_C(                   0), -INT64_C( 5122095061723301309),  INT64_C(                   0),
         INT64_C( 1696414226036203023),  INT64_C( 3296580388577313996),  INT64_C(                   0),  INT64_C(                   0) } },
    { UINT8_C( 94),
      {  INT64_C( 6996533585489046842),  INT64_C( 1958416866463124758),  INT64_C( 4537104879152098493),  INT64_C( 6668091767506160517),
         INT64_C( 8690684359556707327), -INT64_C( 5373299068551870832), -INT64_C( 8523060076547933024),  INT64_C( 8344112879438766413) },
      {  INT64_C( 5389482443695536114), -INT64_C( 6396224148588342641), -INT64_C( 7523587219716487416),  INT64_C( 8890901203756993403),
        -INT64_C( 1724708946921451509), -INT64_C( 4532121262000984723), -INT64_C( 8010949269784740878), -INT64_C( 5137233140829398353) },
      {  INT64_C(                   0), -INT64_C( 4894922666030997607), -INT64_C( 6241477190632068171),  INT64_C( 2876394624261161214),
        -INT64_C( 8031297777887304716),  INT64_C(                   0),  INT64_C( 1831669897844082514),  INT64_C(                   0) } },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i a = simde_mm512_loadu_epi64(test_vec[i].a);
    simde__m512i b = simde_mm512_loadu_epi64(test_vec[i].b);
    simde__m512i r = simde_mm512_castpd_si512(simde_mm512_maskz_xor_pd(test_vec[i].k, simde_mm512_castsi512_pd(a), simde_mm512_castsi512_pd(b)));
    simde_test_x86_assert_equal_i64x8(r, simde_mm512_loadu_epi64(test_vec[i].r));
  }

  return 0;
#else
  fputc('\n', stdout);
  for (int i = 0 ; i < 8 ; i++) {
    simde__mmask8 k = simde_test_x86_random_mmask8();
    simde__m512i a = simde_test_x86_random_i64x8();
    simde__m512i b = simde_test_x86_random_i64x8();
    simde__m512i r = simde_mm512_castpd_si512(simde_mm512_maskz_xor_pd(k, simde_mm512_castsi512_pd(a), simde_mm512_castsi512_pd(b)));

    simde_test_x86_write_mmask8(2, k, SIMDE_TEST_VEC_POS_FIRST);
    simde_test_x86_write_i64x8(2, a, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i64x8(2, b, SIMDE_TEST_VEC_POS_MIDDLE);
    simde_test_x86_write_i64x8(2, r, SIMDE_TEST_VEC_POS_LAST);
  }
  return 1;
#endif
}

static int
test_simde_mm512_xor_epi32 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int32_t a[16];
    const int32_t b[16];
    const int32_t r[16];
  } test_vec[] = {
    { {  INT32_C(   116835021),  INT32_C(   228055744), -INT32_C(   583287396), -INT32_C(  1948084724), -INT32_C(   539666230), -INT32_C(    47767560),  INT32_C(   757520745), -INT32_C(   166532568),
         INT32_C(  1828456620),  INT32_C(  2138739938),  INT32_C(  1566356817),  INT32_C(   384384587), -INT32_C(  1879720297),  INT32_C(  1200365021),  INT32_C(   527741942),  INT32_C(  1226147485) },
      {  INT32_C(  1907757455), -INT32_C(   135253851),  INT32_C(   810831333),  INT32_C(   575028619), -INT32_C(   642630405),  INT32_C(  1327513177), -INT32_C(  1905355535), -INT32_C(  1445428454),
         INT32_C(   974884501), -INT32_C(  1557066818), -INT32_C(   455899560), -INT32_C(  1090118973), -INT32_C(  1348945834), -INT32_C(   385959689),  INT32_C(  1719102796), -INT32_C(  2062528528) },
      {  INT32_C(  2000737090), -INT32_C(    93853083), -INT32_C(   311431047), -INT32_C(  1448825977),  INT32_C(   107439153), -INT32_C(  1308155487), -INT32_C(  1555542632),  INT32_C(  1607100722),
         INT32_C(  1458013753), -INT32_C(   599023780), -INT32_C(  1181797623), -INT32_C(  1443945336),  INT32_C(   544016065), -INT32_C(  1351441110),  INT32_C(  2030296762), -INT32_C(   872035987) } },
    { { -INT32_C(  1681904675), -INT32_C(  1891700426),  INT32_C(   997396855), -INT32_C(  2114291413),  INT32_C(   691049010), -INT32_C(  1777258678), -INT32_C(  1929607012), -INT32_C(  1273885481),
         INT32_C(  1850724919),  INT32_C(  1006472643), -INT32_C(   898207585),  INT32_C(   457994473),  INT32_C(  1279556610),  INT32_C(  1206015659), -INT32_C(  1227563298),  INT32_C(   577431275) },
      {  INT32_C(  2073082552), -INT32_C(   424243897), -INT32_C(   407819011), -INT32_C(  1643971172),  INT32_C(   636110713),  INT32_C(  2070727837), -INT32_C(  1775157077), -INT32_C(   558326746),
        -INT32_C(  1655027370), -INT32_C(   729542442), -INT32_C(   642042563), -INT32_C(  1418150606), -INT32_C(  1580178940), -INT32_C(   635618001), -INT32_C(  1536143747),  INT32_C(  1082271978) },
      { -INT32_C(   531590811),  INT32_C(  1770552433), -INT32_C(   591250038),  INT32_C(   536380599),  INT32_C(   215668043), -INT32_C(   310516777),  INT32_C(   449693751),  INT32_C(  1789563121),
        -INT32_C(   216687775), -INT32_C(   277242603),  INT32_C(   332219810), -INT32_C(  1338716709), -INT32_C(   309060090), -INT32_C(  1644205180),  INT32_C(   312774819),  INT32_C(  1659424257) } },
    { {  INT32_C(  1222564977),  INT32_C(   706503405), -INT32_C(   922495081), -INT32_C(  1720419436),  INT32_C(   221922782), -INT32_C(     1615998), -INT32_C(  1868343386), -INT32_C(   237951361),
        -INT32_C(   281432318), -INT32_C(  1474734831), -INT32_C(  1066328788),  INT32_C(  1985602968), -INT32_C(  1400662998), -INT32_C(  1850971413),  INT32_C(  1092767681),  INT32_C(  2016605045) },
      { -INT32_C(  1285067870), -INT32_C(   329547328),  INT32_C(   917294238), -INT32_C(   592640335), -INT32_C(  2054672230),  INT32_C(  1545024667), -INT32_C(   107136636), -INT32_C(   831402196),
        -INT32_C(    92153542), -INT32_C(   119088038),  INT32_C(  1496224680),  INT32_C(   859167641), -INT32_C(  1481064948),  INT32_C(  1979961330),  INT32_C(   879796488), -INT32_C(  1425874576) },
      { -INT32_C(    71714861), -INT32_C(   968367315), -INT32_C(     5301495),  INT32_C(  1171881253), -INT32_C(  2001570492), -INT32_C(  1544527079),  INT32_C(  1765699106),  INT32_C(  1067510099),
         INT32_C(   364410424),  INT32_C(  1358924107), -INT32_C(  1721790844),  INT32_C(  1164721665),  INT32_C(   188426790), -INT32_C(   407919335),  INT32_C(  1968369353), -INT32_C(   751824379) } },
    { {  INT32_C(   346391482),  INT32_C(   118262879), -INT32_C(  1201587425),  INT32_C(   585864726),  INT32_C(  1204396884),  INT32_C(  2076036467), -INT32_C(   558879378), -INT32_C(   913722865),
        -INT32_C(  1780601034), -INT32_C(   627184965),  INT32_C(  1016266277), -INT32_C(   379683436), -INT32_C(  1825560544),  INT32_C(  1661988341),  INT32_C(   708951835), -INT32_C(  1477194895) },
      { -INT32_C(  1254239750), -INT32_C(   510666052),  INT32_C(  1813848536), -INT32_C(  1101693794), -INT32_C(  1722645084), -INT32_C(  1896062605), -INT32_C(  1850196704),  INT32_C(    37333768),
         INT32_C(   968324733),  INT32_C(   672810576),  INT32_C(    93599847),  INT32_C(  1489300148), -INT32_C(   487516561), -INT32_C(  1754206857),  INT32_C(   858269738),  INT32_C(  1362452948) },
      { -INT32_C(  1583852992), -INT32_C(   427600157), -INT32_C(   730064185), -INT32_C(  1665209720), -INT32_C(   560257296), -INT32_C(   180245504),  INT32_C(  1325928526), -INT32_C(   877651705),
        -INT32_C(  1402382005), -INT32_C(   225989397),  INT32_C(   956745282), -INT32_C(  1315268832),  INT32_C(  1908489807), -INT32_C(   193003390),  INT32_C(   426350385), -INT32_C(   154752347) } },
    { {  INT32_C(   663416279), -INT32_C(  1689279437), -INT32_C(  1868503844),  INT32_C(  1038640334), -INT32_C(   249505414), -INT32_C(   242708282), -INT32_C(  1926975047), -INT32_C(   354526958),
         INT32_C(  2047961158), -INT32_C(   384474868),  INT32_C(   326743365), -INT32_C(  1806671334),  INT32_C(    42364987), -INT32_C(  1158476287), -INT32_C(   767092801), -INT32_C(  1212406416) },
      { -INT32_C(  1724789363),  INT32_C(  1937917486),  INT32_C(   361168123), -INT32_C(  1716856994),  INT32_C(  1218129991), -INT32_C(    33386946),  INT32_C(   382749093), -INT32_C(    70415250),
        -INT32_C(  2003435943),  INT32_C(  1073420100),  INT32_C(  1901429267), -INT32_C(  1609826471),  INT32_C(  1844028975), -INT32_C(   630461643), -INT32_C(  1544537291),  INT32_C(   564116935) },
      { -INT32_C(  1095032742), -INT32_C(   389160419), -INT32_C(  2061101017), -INT32_C(  1539132528), -INT32_C(  1178867395),  INT32_C(   260710136), -INT32_C(  1678444516),  INT32_C(   286512508),
        -INT32_C(   226191841), -INT32_C(   689015224),  INT32_C(  1647064918),  INT32_C(   878419267),  INT32_C(  1869600276),  INT32_C(  1620632884),  INT32_C(  1907829898), -INT32_C(  1776052041) } },
    { { -INT32_C(     5688133),  INT32_C(  1598006347),  INT32_C(  2144375846), -INT32_C(  1038034029), -INT32_C(  1221654142),  INT32_C(   697408500), -INT32_C(  1630698794), -INT32_C(    71340993),
        -INT32_C(   335910752),  INT32_C(   860502284),  INT32_C(  1622285261),  INT32_C(  2015548150), -INT32_C(   802204965), -INT32_C(  1007042067), -INT32_C(  2107521469), -INT32_C(   763551694) },
      { -INT32_C(  1782679416),  INT32_C(  2143815857),  INT32_C(   450853411),  INT32_C(   680657485),  INT32_C(  1106821716), -INT32_C(   972754301), -INT32_C(   347576648),  INT32_C(   264095366),
        -INT32_C(   274433218), -INT32_C(  1485935484),  INT32_C(   868306662), -INT32_C(  1537452976), -INT32_C(  1713023978), -INT32_C(    10491578), -INT32_C(   689264817), -INT32_C(  1411012755) },
      {  INT32_C(  1779911731),  INT32_C(   553102586),  INT32_C(  1695542789), -INT32_C(   357377570), -INT32_C(   153629738), -INT32_C(   275420809),  INT32_C(  1971709038), -INT32_C(   201151815),
         INT32_C(    73274270), -INT32_C(  1809558136),  INT32_C(  1400067371), -INT32_C(   595687770),  INT32_C(  1237976781),  INT32_C(  1017522347),  INT32_C(  1418420492),  INT32_C(  2040039263) } },
    { { -INT32_C(  1482979037), -INT32_C(   615642635), -INT32_C(  1492185001),  INT32_C(  2051763044),  INT32_C(   101920959),  INT32_C(  1761964570), -INT32_C(  2025853159), -INT32_C(  1187896170),
        -INT32_C(  1570714195),  INT32_C(   780054487),  INT32_C(   601263551), -INT32_C(  1214438920),  INT32_C(  1824370770),  INT32_C(  1003864610),  INT32_C(  1220679089), -INT32_C(   419302087) },
      { -INT32_C(  1719049534), -INT32_C(   792197359), -INT32_C(  1930191212),  INT32_C(   289706175),  INT32_C(  1652359488),  INT32_C(  1973244868), -INT32_C(  1581424536),  INT32_C(   394837845),
         INT32_C(   850465313), -INT32_C(  1375569639), -INT32_C(   700779241), -INT32_C(   974619003),  INT32_C(  1143432576),  INT32_C(   549045432),  INT32_C(  2042787620),  INT32_C(  1469139510) },
      {  INT32_C(  1041427425),  INT32_C(   193335012),  INT32_C(   737971907),  INT32_C(  1796209627),  INT32_C(  1684943359),  INT32_C(   479732190),  INT32_C(   646090609), -INT32_C(  1363502141),
        -INT32_C(  1865293940), -INT32_C(  2139302194), -INT32_C(   169051992),  INT32_C(  1920294781),  INT32_C(   681235922),  INT32_C(   460064410),  INT32_C(   822108821), -INT32_C(  1332691185) } },
    { {  INT32_C(  1988706908), -INT32_C(   769356869),  INT32_C(    94920320),  INT32_C(  1573556445), -INT32_C(  1365118474), -INT32_C(   623945035),  INT32_C(   122917329),  INT32_C(   945743067),
        -INT32_C(   508631258), -INT32_C(   223096206),  INT32_C(   234314800), -INT32_C(   496320020),  INT32_C(  1754336178),  INT32_C(   927096934), -INT32_C(   868248079), -INT32_C(  1610310278) },
      { -INT32_C(   142429563), -INT32_C(  1242942076),  INT32_C(  2126700945),  INT32_C(  1432366499), -INT32_C(  1631719112),  INT32_C(  1121386321),  INT32_C(   252646805),  INT32_C(   934220722),
         INT32_C(  1244606918), -INT32_C(   117499545), -INT32_C(  1653161222),  INT32_C(   670291951),  INT32_C(   449229000),  INT32_C(  1163697328),  INT32_C(  1666476977),  INT32_C(  1167786879) },
      { -INT32_C(  2129923879),  INT32_C(  1741602367),  INT32_C(  2070593297),  INT32_C(   145407358),  INT32_C(   807141582), -INT32_C(  1743149596),  INT32_C(   140280900),  INT32_C(   267515753),
        -INT32_C(  1417554208),  INT32_C(   172804885), -INT32_C(  1870553398), -INT32_C(   979823101),  INT32_C(  1918352250),  INT32_C(  1914633430), -INT32_C(  1351877056), -INT32_C(   442524155) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i a = simde_mm512_loadu_epi32(test_vec[i].a);
    simde__m512i b = simde_mm512_loadu_epi32(test_vec[i].b);
    simde__m512i r = simde_mm512_xor_epi32(a, b);
    simde_test_x86_assert_equal_i32x16(r, simde_mm512_loadu_epi32(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm512_mask_xor_epi32 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int32_t src[16];
    const simde__mmask16 k;
    const int32_t a[16];
    const int32_t b[16];
    const int32_t r[16];
  } test_vec[] = {
    { {  INT32_C(   153880592), -INT32_C(  1121219888),  INT32_C(   485857388), -INT32_C(   948181383),  INT32_C(    41449270), -INT32_C(  1756100706),  INT32_C(  1579137812),  INT32_C(   587505939),
         INT32_C(  2049716394),  INT32_C(   641161146),  INT32_C(  1765944559),  INT32_C(  1211153938), -INT32_C(   733304523), -INT32_C(   999579984),  INT32_C(  1814202969), -INT32_C(   695261652) },
      UINT16_C(47959),
      { -INT32_C(  2028859056),  INT32_C(  2058551864), -INT32_C(  1690778261),  INT32_C(  1464102413), -INT32_C(  1376390078), -INT32_C(   617066567),  INT32_C(  1258448059), -INT32_C(  1979295430),
        -INT32_C(  1575937686), -INT32_C(  2044869605), -INT32_C(  1776200311),  INT32_C(    99444419),  INT32_C(   330555994), -INT32_C(   303043791), -INT32_C(  1992756913), -INT32_C(  1273807286) },
      {  INT32_C(  1918313559),  INT32_C(  1912108264), -INT32_C(  1928914231), -INT32_C(   627903360),  INT32_C(   149833174),  INT32_C(  2146819376),  INT32_C(   403189198), -INT32_C(  1010033813),
         INT32_C(   657793855),  INT32_C(  1637363351), -INT32_C(   923885752),  INT32_C(  1789034643), -INT32_C(   160263994),  INT32_C(   997615469), -INT32_C(    11305324), -INT32_C(   641589094) },
      { -INT32_C(   180013817),  INT32_C(   189494992),  INT32_C(   373284770), -INT32_C(   948181383), -INT32_C(  1524911724), -INT32_C(  1756100706),  INT32_C(  1393183093),  INT32_C(   587505939),
        -INT32_C(  2061223339), -INT32_C(   410654068),  INT32_C(  1765944559),  INT32_C(  1867507280), -INT32_C(   440307044), -INT32_C(   694580132),  INT32_C(  1814202969),  INT32_C(  1842421456) } },
    { { -INT32_C(   620627901),  INT32_C(  1849465126), -INT32_C(   868865479),  INT32_C(  1882642602), -INT32_C(   697849751), -INT32_C(  1559044849), -INT32_C(   173841061), -INT32_C(   909154939),
        -INT32_C(  2086350755), -INT32_C(  1561206679), -INT32_C(  1267783926),  INT32_C(  1764074752),  INT32_C(  1564445774), -INT32_C(  1006546583),  INT32_C(  1018864823),  INT32_C(  1728416009) },
      UINT16_C(43353),
      { -INT32_C(   594951446), -INT32_C(   737963163),  INT32_C(  1836647240), -INT32_C(  1392916628),  INT32_C(   654139941), -INT32_C(   523586522), -INT32_C(   160836623),  INT32_C(   631292474),
        -INT32_C(   369022587),  INT32_C(    96339133),  INT32_C(  1953642248),  INT32_C(   606104575), -INT32_C(   213246259), -INT32_C(  1009511214),  INT32_C(   582630887), -INT32_C(  2058921472) },
      {  INT32_C(  1081034883),  INT32_C(  1413819724),  INT32_C(  1674164068), -INT32_C(   276305630), -INT32_C(   622604025), -INT32_C(   845301787), -INT32_C(   185640972),  INT32_C(   897136305),
        -INT32_C(   898242434),  INT32_C(  2048899861), -INT32_C(  1780619150), -INT32_C(   645634607),  INT32_C(   481519414),  INT32_C(   334057502),  INT32_C(  1493686440), -INT32_C(  1936817906) },
      { -INT32_C(  1662612887),  INT32_C(  1849465126), -INT32_C(   868865479),  INT32_C(  1132342862), -INT32_C(    65096926), -INT32_C(  1559044849),  INT32_C(    42370053), -INT32_C(   909154939),
         INT32_C(   544522747), -INT32_C(  1561206679), -INT32_C(  1267783926), -INT32_C(    39580114),  INT32_C(  1564445774), -INT32_C(   801291060),  INT32_C(  1018864823),  INT32_C(   164223758) } },
    { {  INT32_C(  2136343657),  INT32_C(   838432191),  INT32_C(   801560157),  INT32_C(  1913146171), -INT32_C(   779174990), -INT32_C(  1276872949),  INT32_C(  1561193295), -INT32_C(   689333396),
         INT32_C(  1582645151),  INT32_C(   311447221),  INT32_C(  1614894628),  INT32_C(  1423067553),  INT32_C(   270884868),  INT32_C(   650316247),  INT32_C(  1636028660),  INT32_C(   188181612) },
      UINT16_C(36011),
      { -INT32_C(    86351766), -INT32_C(  1269760397),  INT32_C(   822014558),  INT32_C(  1804599878), -INT32_C(   697014510),  INT32_C(   279341197),  INT32_C(     8131273), -INT32_C(  2004080610),
        -INT32_C(    75340152), -INT32_C(  1011821979),  INT32_C(   183807684), -INT32_C(  1015708496),  INT32_C(  2040130028),  INT32_C(   462045010),  INT32_C(  1880884818), -INT32_C(  1208440786) },
      {  INT32_C(  1957853711),  INT32_C(   288842316), -INT32_C(  1055184112), -INT32_C(  1669033552), -INT32_C(   871031430), -INT32_C(  1360486564), -INT32_C(   736230234), -INT32_C(  1148512596),
        -INT32_C(   584106352), -INT32_C(  1309775968),  INT32_C(  1114769810),  INT32_C(   366933658),  INT32_C(  1877079059),  INT32_C(   958318995),  INT32_C(  2047753421), -INT32_C(   483026605) },
      { -INT32_C(  1905780123), -INT32_C(  1519936449),  INT32_C(   801560157), -INT32_C(   149646346), -INT32_C(   779174990), -INT32_C(  1102121007),  INT32_C(  1561193295),  INT32_C(   856112818),
         INT32_C(  1582645151),  INT32_C(   311447221),  INT32_C(  1216784214), -INT32_C(   693407190),  INT32_C(   270884868),  INT32_C(   650316247),  INT32_C(  1636028660),  INT32_C(  1422733693) } },
    { {  INT32_C(  2025874903),  INT32_C(  1579790028),  INT32_C(  1386322872), -INT32_C(  1536721007),  INT32_C(   118704499), -INT32_C(   532664046), -INT32_C(  1051046290), -INT32_C(  1079734296),
        -INT32_C(  1070111244), -INT32_C(   887136237), -INT32_C(  1944141573), -INT32_C(  1288665793), -INT32_C(   507886386), -INT32_C(   473826699),  INT32_C(   832838473), -INT32_C(  1611642454) },
      UINT16_C(10414),
      {  INT32_C(  2122891615), -INT32_C(  1421966195),  INT32_C(  1093697040), -INT32_C(   343539919), -INT32_C(  1578763552), -INT32_C(  2101596450),  INT32_C(  1355507295),  INT32_C(  1702394117),
        -INT32_C(   924647365), -INT32_C(  1787616636), -INT32_C(   774462560), -INT32_C(  2101584990),  INT32_C(   857973077),  INT32_C(   800514000),  INT32_C(  1266647366),  INT32_C(   917567483) },
      {  INT32_C(  2097059064),  INT32_C(  1443983798), -INT32_C(  1222121708), -INT32_C(  1741036734),  INT32_C(  1422613892), -INT32_C(  2105245380), -INT32_C(    20053245), -INT32_C(   214663429),
        -INT32_C(   932236782), -INT32_C(  1222737501), -INT32_C(  1418836632), -INT32_C(  1388074968),  INT32_C(  1090588165), -INT32_C(  1815837296), -INT32_C(  2070834551),  INT32_C(   595051792) },
      {  INT32_C(  2025874903), -INT32_C(    47188677), -INT32_C(   166225660),  INT32_C(  1941707891),  INT32_C(   118704499),  INT32_C(     3715042), -INT32_C(  1051046290), -INT32_C(  1773402114),
        -INT32_C(  1070111244), -INT32_C(   887136237), -INT32_C(  1944141573),  INT32_C(   805303178), -INT32_C(   507886386), -INT32_C(  1133356480),  INT32_C(   832838473), -INT32_C(  1611642454) } },
    { { -INT32_C(  1695815945), -INT32_C(   799995289),  INT32_C(  2021376079),  INT32_C(  1831190120),  INT32_C(  1554917068),  INT32_C(   904884908),  INT32_C(   364478468),  INT32_C(  1010315333),
         INT32_C(  2127962902),  INT32_C(  2102274093),  INT32_C(  1358285288),  INT32_C(  1404902023), -INT32_C(   323982528), -INT32_C(   501112866),  INT32_C(  1710742048),  INT32_C(   564277002) },
      UINT16_C(30803),
      { -INT32_C(   308248417), -INT32_C(   222918403), -INT32_C(  1794359848),  INT32_C(  1073827216),  INT32_C(  1541463865), -INT32_C(  1187708735),  INT32_C(   132661349),  INT32_C(     8403809),
        -INT32_C(  1175641924), -INT32_C(  2119392343),  INT32_C(  1880602848),  INT32_C(  1068570629), -INT32_C(  1197829641), -INT32_C(   160313455),  INT32_C(  1895717391),  INT32_C(  1366326933) },
      {  INT32_C(  1191927198), -INT32_C(   523651328),  INT32_C(  1968300143), -INT32_C(   273415432),  INT32_C(   614944403),  INT32_C(   756685085),  INT32_C(   144513139),  INT32_C(   895094422),
         INT32_C(  1820091755), -INT32_C(  1957935844),  INT32_C(   486579493),  INT32_C(   839693471),  INT32_C(   542553090),  INT32_C(  1078817229),  INT32_C(   541649545),  INT32_C(  1683333880) },
      { -INT32_C(  1431577343),  INT32_C(   310329341),  INT32_C(  2021376079),  INT32_C(  1831190120),  INT32_C(  2135396778),  INT32_C(   904884908),  INT32_C(   259348502),  INT32_C(  1010315333),
         INT32_C(  2127962902),  INT32_C(  2102274093),  INT32_C(  1358285288),  INT32_C(   230534298), -INT32_C(  1731451403), -INT32_C(  1237533092),  INT32_C(  1354150022),  INT32_C(   564277002) } },
    { {  INT32_C(   600887559),  INT32_C(  1018108951),  INT32_C(  1499115450),  INT32_C(  1720477284), -INT32_C(   410590694), -INT32_C(   584592557), -INT32_C(  1224904514),  INT32_C(   437932562),
         INT32_C(   977136163), -INT32_C(  1049105401),  INT32_C(     1757596),  INT32_C(  1382458935), -INT32_C(   600183415),  INT32_C(  2126078400), -INT32_C(   483019055),  INT32_C(   754798344) },
      UINT16_C(15162),
      { -INT32_C(   584629914),  INT32_C(   481215234),  INT32_C(   717481667),  INT32_C(  1897352248), -INT32_C(   489498840), -INT32_C(  1952930986), -INT32_C(  2066046585),  INT32_C(   868160717),
         INT32_C(  1460791125),  INT32_C(  1853145002), -INT32_C(   577226843), -INT32_C(  1420841085), -INT32_C(   594730618),  INT32_C(  1281828549), -INT32_C(  1798290745), -INT32_C(  1396207530) },
      {  INT32_C(   553900151),  INT32_C(  1032812440),  INT32_C(   823863214),  INT32_C(  1574791894),  INT32_C(  1345940107),  INT32_C(  1469882768),  INT32_C(   971730403),  INT32_C(  1961210877),
         INT32_C(   597027211),  INT32_C(   258024800),  INT32_C(   591428684),  INT32_C(  1904221670),  INT32_C(   415414664),  INT32_C(  1030774362), -INT32_C(   931701813), -INT32_C(  1707320306) },
      {  INT32_C(   600887559),  INT32_C(   555857050),  INT32_C(  1499115450),  INT32_C(   751445742), -INT32_C(  1293173341), -INT32_C(   603716922), -INT32_C(  1224904514),  INT32_C(   437932562),
         INT32_C(  1954811614),  INT32_C(  1628805834),  INT32_C(     1757596), -INT32_C(   623922587), -INT32_C(  1001416690),  INT32_C(  1897363615), -INT32_C(   483019055),  INT32_C(   754798344) } },
    { { -INT32_C(  1497509307),  INT32_C(  1135943415), -INT32_C(  2140736102), -INT32_C(  1678580205), -INT32_C(    88886112), -INT32_C(   566746350), -INT32_C(  1918455937),  INT32_C(  1361568523),
        -INT32_C(  1393039947), -INT32_C(  1628460029), -INT32_C(  1273080159), -INT32_C(   598798276), -INT32_C(   673774652), -INT32_C(  1514860762), -INT32_C(   919381058), -INT32_C(   216376770) },
      UINT16_C( 4415),
      { -INT32_C(  1883421793), -INT32_C(     1744927),  INT32_C(  1662001427), -INT32_C(   714681090),  INT32_C(  1625590955),  INT32_C(  1706861106), -INT32_C(  2034238869), -INT32_C(  1919418386),
         INT32_C(   589059394), -INT32_C(   954072652),  INT32_C(   556413475), -INT32_C(  1292464121),  INT32_C(  1326701085), -INT32_C(   407580804), -INT32_C(  1217563703), -INT32_C(  1270545038) },
      {  INT32_C(   248996186), -INT32_C(  2032797085),  INT32_C(   866582572), -INT32_C(  1377395312), -INT32_C(   201524873), -INT32_C(  1847938872), -INT32_C(  1773582300), -INT32_C(  1488220595),
         INT32_C(  1387602671),  INT32_C(  1238927900),  INT32_C(   461143946), -INT32_C(  1815584228),  INT32_C(   596034907), -INT32_C(  1716166283), -INT32_C(   164561239),  INT32_C(  2073918348) },
      { -INT32_C(  2123750715),  INT32_C(  2033231234),  INT32_C(  1354178879),  INT32_C(  2021673326), -INT32_C(  1827113508), -INT32_C(   194637062), -INT32_C(  1918455937),  INT32_C(  1361568523),
         INT32_C(  1906931629), -INT32_C(  1628460029), -INT32_C(  1273080159), -INT32_C(   598798276),  INT32_C(  1821712198), -INT32_C(  1514860762), -INT32_C(   919381058), -INT32_C(   216376770) } },
    { { -INT32_C(  1160883555),  INT32_C(  1728292572),  INT32_C(  1115848486),  INT32_C(  1037454050), -INT32_C(  2057217009),  INT32_C(  1713247933), -INT32_C(  1604563436),  INT32_C(  1729886665),
         INT32_C(   673311051), -INT32_C(  1232132976), -INT32_C(  2047340125),  INT32_C(  1807994459), -INT32_C(   386915285),  INT32_C(  1313803834),  INT32_C(   653175645), -INT32_C(   259192411) },
      UINT16_C(44787),
      { -INT32_C(  1479376104),  INT32_C(   834172473), -INT32_C(  1107290885),  INT32_C(  1877093247),  INT32_C(  1652366355),  INT32_C(  1477368426), -INT32_C(  1906134271), -INT32_C(  1136831069),
         INT32_C(   291704792), -INT32_C(  2126373755), -INT32_C(  1354808784), -INT32_C(  2128731539), -INT32_C(  1444701378),  INT32_C(  1996616054),  INT32_C(  1208312740), -INT32_C(  1878768968) },
      { -INT32_C(   677288110), -INT32_C(  1286019965), -INT32_C(  1805477850), -INT32_C(   132808518), -INT32_C(  1834878948), -INT32_C(  1911970838), -INT32_C(  1076490746), -INT32_C(  1555047855),
        -INT32_C(   981798590), -INT32_C(    75901995),  INT32_C(   630184811),  INT32_C(  2015208540), -INT32_C(  2029338723),  INT32_C(  1746211682),  INT32_C(  1915218977),  INT32_C(   135624646) },
      {  INT32_C(  1886643274), -INT32_C(  2099211590),  INT32_C(  1115848486),  INT32_C(  1037454050), -INT32_C(   253959153), -INT32_C(   704153216),  INT32_C(   833928455),  INT32_C(   527601650),
         INT32_C(   673311051),  INT32_C(  2050740048), -INT32_C(  1968137893), -INT32_C(   117209551), -INT32_C(   386915285),  INT32_C(   521462292),  INT32_C(   653175645), -INT32_C(  1743702658) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i src = simde_mm512_loadu_epi32(test_vec[i].src);
    simde__m512i a = simde_mm512_loadu_epi32(test_vec[i].a);
    simde__m512i b = simde_mm512_loadu_epi32(test_vec[i].b);
    simde__m512i r = simde_mm512_mask_xor_epi32(src, test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i32x16(r, simde_mm512_loadu_epi32(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm512_maskz_xor_epi32 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde__mmask16 k;
    const int32_t a[16];
    const int32_t b[16];
    const int32_t r[16];
  } test_vec[] = {
    { UINT16_C(31138),
      { -INT32_C(   268230814), -INT32_C(   127828744), -INT32_C(   652244455), -INT32_C(   178928494), -INT32_C(  1171138727),  INT32_C(  1381371835),  INT32_C(  1040805621), -INT32_C(   726160526),
        -INT32_C(  1027294518),  INT32_C(  1337664822),  INT32_C(  1093196463), -INT32_C(   147358050),  INT32_C(   296904789),  INT32_C(  1818429559), -INT32_C(   374707081), -INT32_C(   507682537) },
      {  INT32_C(  1386512668),  INT32_C(  1453481894), -INT32_C(   661140935), -INT32_C(  1630548408), -INT32_C(  1364229833),  INT32_C(    18551434), -INT32_C(  1762998913),  INT32_C(  1131980583),
        -INT32_C(   829088728), -INT32_C(  1272694917),  INT32_C(  1250737154), -INT32_C(  1041736566),  INT32_C(  1735432157),  INT32_C(   694717354),  INT32_C(  2009027152),  INT32_C(   565852153) },
      {  INT32_C(           0), -INT32_C(  1362943138),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(  1397562673),  INT32_C(           0), -INT32_C(  1748038571),
         INT32_C(   206677730),  INT32_C(           0),  INT32_C(           0),  INT32_C(   920592916),  INT32_C(  1992490888),  INT32_C(  1158382557), -INT32_C(  1642775001),  INT32_C(           0) } },
    { UINT16_C(20563),
      {  INT32_C(   344444656),  INT32_C(   265324931), -INT32_C(  1133814828), -INT32_C(  1940633316),  INT32_C(   421002928), -INT32_C(   429168601), -INT32_C(  1709283873),  INT32_C(  1995075974),
        -INT32_C(  1014271680), -INT32_C(   808297477), -INT32_C(   745849162),  INT32_C(   929030023),  INT32_C(    72382429), -INT32_C(  1091846945), -INT32_C(  1487402719), -INT32_C(  1172422022) },
      { -INT32_C(  1350719052), -INT32_C(  1149349884),  INT32_C(   361630094), -INT32_C(   968036887),  INT32_C(  1137417572),  INT32_C(  2046998105),  INT32_C(   958487231),  INT32_C(  1374961565),
        -INT32_C(   318737944),  INT32_C(  1336442561),  INT32_C(  1885615751), -INT32_C(  2009616093), -INT32_C(  1479867826),  INT32_C(  1998704056), -INT32_C(   978304472),  INT32_C(  1779868802) },
      { -INT32_C(  1141217468), -INT32_C(  1263609465),  INT32_C(           0),  INT32_C(           0),  INT32_C(  1524392916),  INT32_C(           0), -INT32_C(  1556136096),  INT32_C(           0),
         INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0), -INT32_C(  1550092909),  INT32_C(           0),  INT32_C(  1659390729),  INT32_C(           0) } },
    { UINT16_C( 5654),
      { -INT32_C(     7022761), -INT32_C(  1959453913), -INT32_C(  1019455092), -INT32_C(  1396339999),  INT32_C(  1383759153), -INT32_C(  1533697292),  INT32_C(  2101942119), -INT32_C(   644653183),
         INT32_C(  1574446902), -INT32_C(   823587262),  INT32_C(  1217471847), -INT32_C(   504080720), -INT32_C(   952865069),  INT32_C(  2003618064),  INT32_C(  1643427296),  INT32_C(  1245349652) },
      { -INT32_C(   240643154), -INT32_C(  2000711391),  INT32_C(  1724928438),  INT32_C(  2068366503),  INT32_C(  1128430643),  INT32_C(   633056837),  INT32_C(  2005315427), -INT32_C(   440221385),
        -INT32_C(   170497324), -INT32_C(  1317169413), -INT32_C(  1911009817),  INT32_C(  1141465105),  INT32_C(   562580700),  INT32_C(  1581663226),  INT32_C(   701877490),  INT32_C(  1628411789) },
      {  INT32_C(           0),  INT32_C(    59410950), -INT32_C(  1511258054),  INT32_C(           0),  INT32_C(   288882946),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),
         INT32_C(           0),  INT32_C(  2140444857), -INT32_C(   964073344),  INT32_C(           0), -INT32_C(   423878129),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0) } },
    { UINT16_C(58625),
      { -INT32_C(   746849194), -INT32_C(   970956114), -INT32_C(    98159887), -INT32_C(    28966282),  INT32_C(  1765884195),  INT32_C(  1949643935),  INT32_C(  1812775773),  INT32_C(  2068974884),
        -INT32_C(  1202795254),  INT32_C(   561934128),  INT32_C(   387687585), -INT32_C(   921279834),  INT32_C(  1110595491), -INT32_C(   373921909),  INT32_C(  1331020330), -INT32_C(   624252976) },
      { -INT32_C(  1517152139),  INT32_C(   684068999),  INT32_C(  1514201524), -INT32_C(   433891773),  INT32_C(   958944685), -INT32_C(   417144900),  INT32_C(  1899394977), -INT32_C(  1807024097),
        -INT32_C(  1606820584), -INT32_C(  1580662803),  INT32_C(   620431584),  INT32_C(   201989726),  INT32_C(   809841523), -INT32_C(  1290311918), -INT32_C(    47886882),  INT32_C(  1704030285) },
      {  INT32_C(  1995039779),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),
         INT32_C(   410456082),  INT32_C(           0),  INT32_C(   870362177),  INT32_C(           0),  INT32_C(           0),  INT32_C(  1520500889), -INT32_C(  1301245964), -INT32_C(  1084500067) } },
    { UINT16_C(51790),
      { -INT32_C(   825672955), -INT32_C(   657020195), -INT32_C(   654887474), -INT32_C(  2046072255), -INT32_C(  1309860454), -INT32_C(   151073839),  INT32_C(  1516653513), -INT32_C(  1239042895),
        -INT32_C(   846926096),  INT32_C(  1722112920), -INT32_C(   784360304), -INT32_C(  1588114937),  INT32_C(   961758312), -INT32_C(   667987697),  INT32_C(  1312003485),  INT32_C(   973363274) },
      { -INT32_C(   553154234),  INT32_C(  1950723300),  INT32_C(  1346733129),  INT32_C(   938581199), -INT32_C(   277855008),  INT32_C(   868786070),  INT32_C(  2122447668), -INT32_C(  1715960493),
        -INT32_C(   226967794), -INT32_C(  1251557780),  INT32_C(   285584194),  INT32_C(   659093063), -INT32_C(   786974661), -INT32_C(  1945837737),  INT32_C(   755664346),  INT32_C(   432456458) },
      {  INT32_C(           0), -INT32_C(  1399650759), -INT32_C(  2001555065), -INT32_C(  1308952946),  INT32_C(           0),  INT32_C(           0),  INT32_C(   619163901),  INT32_C(           0),
         INT32_C(           0), -INT32_C(   742136332),  INT32_C(           0), -INT32_C(  2044740544),  INT32_C(           0),  INT32_C(           0),  INT32_C(  1664684103),  INT32_C(   599956288) } },
    { UINT16_C(16258),
      {  INT32_C(  1929244171), -INT32_C(  1474478173), -INT32_C(  1717607087), -INT32_C(  1554851188), -INT32_C(  1350391381),  INT32_C(  1077173301),  INT32_C(  1342389897),  INT32_C(  1670350167),
         INT32_C(   399871092),  INT32_C(   482407115), -INT32_C(   457810089), -INT32_C(   460912583), -INT32_C(   409793871), -INT32_C(   265828506),  INT32_C(  1564486149),  INT32_C(   599838639) },
      {  INT32_C(   641439067), -INT32_C(   549192825), -INT32_C(  1815873190), -INT32_C(  1300805119), -INT32_C(  1147598252), -INT32_C(   693387055), -INT32_C(  1707873302),  INT32_C(   364770234),
         INT32_C(   272365704),  INT32_C(  1307541235),  INT32_C(  2027991671),  INT32_C(  1361795068),  INT32_C(   839697505),  INT32_C(  1862907781),  INT32_C(  1544109218), -INT32_C(  1200502992) },
      {  INT32_C(           0),  INT32_C(  2002699300),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(  1983018733),
         INT32_C(   133068028),  INT32_C(  1362070584), -INT32_C(  1672024800), -INT32_C(  1246994491), -INT32_C(   710947120), -INT32_C(  1624346397),  INT32_C(           0),  INT32_C(           0) } },
    { UINT16_C(44223),
      { -INT32_C(  1221872696), -INT32_C(   513170944),  INT32_C(  1178101275), -INT32_C(  1022715209), -INT32_C(   713388084),  INT32_C(   135404799), -INT32_C(   372293000), -INT32_C(  1013608454),
         INT32_C(  1115340866),  INT32_C(  2116281443),  INT32_C(    29645898), -INT32_C(  1027289354),  INT32_C(  1570258526), -INT32_C(   765089190), -INT32_C(   423938580),  INT32_C(   111759556) },
      {  INT32_C(  1950884881),  INT32_C(  1391619080), -INT32_C(  1101809976), -INT32_C(   494921852), -INT32_C(  1354819499), -INT32_C(  1367235134), -INT32_C(  1634386726), -INT32_C(  1650180468),
         INT32_C(  1779559522),  INT32_C(   549192536),  INT32_C(  1054740409),  INT32_C(  2065719078),  INT32_C(   975855735), -INT32_C(   538399995),  INT32_C(  1954381287),  INT32_C(   504439228) },
      { -INT32_C(  1016883239), -INT32_C(  1281634808), -INT32_C(   127151917),  INT32_C(   562728243),  INT32_C(  2051381145), -INT32_C(  1500268227),  INT32_C(           0),  INT32_C(  1580314998),
         INT32_C(           0),  INT32_C(           0),  INT32_C(  1058690035), -INT32_C(  1176202800),  INT32_C(           0),  INT32_C(   227344735),  INT32_C(           0),  INT32_C(   414740856) } },
    { UINT16_C( 8973),
      {  INT32_C(  1160144521),  INT32_C(  1700126854),  INT32_C(  1053063966),  INT32_C(   564018167),  INT32_C(  1590535029), -INT32_C(     2378622),  INT32_C(   975214376),  INT32_C(  1063071414),
         INT32_C(   444892052), -INT32_C(  2122327709),  INT32_C(  1287668565), -INT32_C(   194158977), -INT32_C(  2074985982),  INT32_C(   394538735),  INT32_C(  2068948165),  INT32_C(  1723510482) },
      { -INT32_C(  1786691790),  INT32_C(  1830223896), -INT32_C(  1011230908),  INT32_C(   934750005),  INT32_C(  1354500705), -INT32_C(    43499464), -INT32_C(  1216759324), -INT32_C(  1709362328),
        -INT32_C(  1959813518), -INT32_C(   470268257),  INT32_C(  1403433501),  INT32_C(   982146521), -INT32_C(  1618262425),  INT32_C(  1805513606),  INT32_C(   337778348), -INT32_C(  1129431223) },
      { -INT32_C(   794338885),  INT32_C(           0), -INT32_C(    42095526),  INT32_C(   371793090),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),
        -INT32_C(  1851056666),  INT32_C(  1653055484),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(  2082069865),  INT32_C(           0),  INT32_C(           0) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i a = simde_mm512_loadu_epi32(test_vec[i].a);
    simde__m512i b = simde_mm512_loadu_epi32(test_vec[i].b);
    simde__m512i r = simde_mm512_maskz_xor_epi32(test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i32x16(r, simde_mm512_loadu_epi32(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm512_xor_epi64 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int64_t a[8];
    const int64_t b[8];
    const int64_t r[8];
  } test_vec[] = {
    { {  INT64_C( 7833673612016144483), -INT64_C( 6286797061584980339), -INT64_C( 1894311286508114473),  INT64_C( 7457901623545148981),
         INT64_C( 7758400186936267162), -INT64_C( 7827311421232976810),  INT64_C( 5569310173466725242), -INT64_C( 1852634981124308256) },
      { -INT64_C( 6235004628972997840), -INT64_C( 3743688304300620355), -INT64_C( 8561205822545100854),  INT64_C( 2103383255765864446),
        -INT64_C( 2114542116453788245), -INT64_C( 7978576687197911256), -INT64_C( 1001589171339167514), -INT64_C( 1694078873958054286) },
      { -INT64_C( 4193382358810418349),  INT64_C( 7263022192053547824),  INT64_C( 7820062331317703197),  INT64_C( 8813401457401952715),
        -INT64_C( 8571197845464267727),  INT64_C(  151340619297975166), -INT64_C( 4660229538766687332),  INT64_C( 1024413272115446930) } },
    { {  INT64_C( 8415412740149389505),  INT64_C( 8136937495115801748),  INT64_C( 8472142827390787501), -INT64_C(  232688289166536801),
        -INT64_C( 3621830638293505029), -INT64_C( 8157403346233676848), -INT64_C( 8940555774217696396),  INT64_C( 1209819125977694866) },
      {  INT64_C( 3014619373212577232), -INT64_C( 8706215558938034241), -INT64_C( 5583923536013140710), -INT64_C( 3039284310411517779),
        -INT64_C(  181353180207109384),  INT64_C( 7516167267833216947), -INT64_C( 1874932065944318681), -INT64_C( 4710068369810294072) },
      {  INT64_C( 6710250289300899089), -INT64_C(  594058874992207061), -INT64_C( 4101950076832841033),  INT64_C( 2960861935095328562),
         INT64_C( 3514804452886151427), -INT64_C( 1835816431532026781),  INT64_C( 7356107569477813843), -INT64_C( 5879346165530970022) } },
    { { -INT64_C( 1666178802879790940),  INT64_C( 8017248771709566347),  INT64_C( 6370254332653643850),  INT64_C( 3219730174574923307),
        -INT64_C( 3070348517276991929),  INT64_C( 6495017567146349163), -INT64_C( 5439991518983235431), -INT64_C( 5474315090744007085) },
      {  INT64_C( 8012065384011076881),  INT64_C( 7498569038459756453), -INT64_C( 8909788908341826275),  INT64_C( 7194894378406510264),
         INT64_C( 4751279587353037180),  INT64_C( 8288320232820773582),  INT64_C( 7109622136361818100), -INT64_C( 1430348694698849934) },
      { -INT64_C( 8660395485888014923),  INT64_C(  527688101943681582), -INT64_C( 2576724598806067881),  INT64_C( 5726223408186454163),
        -INT64_C( 7742811636499433669),  INT64_C( 2964766335513985189), -INT64_C( 3014254901916840083),  INT64_C( 6350412050110132001) } },
    { { -INT64_C( 6753046675043064470),  INT64_C( 4194897748161412973),  INT64_C( 3956704487311102429),  INT64_C( 4426793051552903443),
        -INT64_C( 6594523062152668411), -INT64_C( 2500730345433285685), -INT64_C( 4041049702590534928),  INT64_C(  851383767448348095) },
      {  INT64_C( 7741231053211505812),  INT64_C( 7420181171167614491),  INT64_C( 7776599092500323953), -INT64_C( 8935477080687233300),
         INT64_C( 6912388001218618338), -INT64_C( 6814099961521935981),  INT64_C( 9145319435102817234), -INT64_C( 7467689775987562262) },
      { -INT64_C( 3952406250137193986),  INT64_C( 6687425642990135670),  INT64_C( 6702779462972608428), -INT64_C( 4714725391518644225),
        -INT64_C(  318013443086337817),  INT64_C( 8945533351659557464), -INT64_C( 5115529358649122526), -INT64_C( 7814353725753193131) } },
    { { -INT64_C(  363657528445588654), -INT64_C( 5313838659546157200), -INT64_C( 6481207919005857538),  INT64_C( 4635854309387370247),
        -INT64_C( 1807691163233783929),  INT64_C( 6833305750270831930),  INT64_C( 3401057919711447504),  INT64_C( 1561413775749067132) },
      { -INT64_C( 8859444360416820816),  INT64_C( 3069616949972669147), -INT64_C(   10721871889266815), -INT64_C( 5170953676681197927),
         INT64_C(  749826453865906582),  INT64_C(  288756504667163261),  INT64_C( 4172734175648855909), -INT64_C(  212340540276977075) },
      {  INT64_C( 9221376615234022114), -INT64_C( 7144947895540580949),  INT64_C( 6473914840305078143), -INT64_C(  546965349575837282),
        -INT64_C( 1401145767694422511),  INT64_C( 6545166095291246407),  INT64_C( 1646764641365311157), -INT64_C( 1682414516424613071) } },
    { {  INT64_C( 6059952658790214623), -INT64_C( 6712846976013345200),  INT64_C( 6538509831123994084), -INT64_C( 7337563655741956666),
        -INT64_C( 6786149137027429721),  INT64_C( 1152722674958794016), -INT64_C( 4236109339647833136),  INT64_C( 5314845287287874925) },
      {  INT64_C( 5484145387571211049), -INT64_C( 6138219535088906346),  INT64_C( 2375857380799177524),  INT64_C( 3786695004706696354),
        -INT64_C( 8939704712816601994),  INT64_C( 8564742865895594328), -INT64_C( 4446283047571463305),  INT64_C( 1643131752719750285) },
      {  INT64_C( 1730191473646272758),  INT64_C(  578577205123273158),  INT64_C( 8810670071267321552), -INT64_C( 5861795896640474780),
         INT64_C( 2467137782418923217),  INT64_C( 8728925404920833144),  INT64_C(  539853474565384359),  INT64_C( 6849847759138550240) } },
    { { -INT64_C( 7108746993929192473),  INT64_C( 2952580909138746433), -INT64_C( 3318415959729659377),  INT64_C( 8351705111930271794),
        -INT64_C( 8460680085403819854),  INT64_C( 2384100826312238396),  INT64_C( 4161117182544079274),  INT64_C( 1385686332839962939) },
      {  INT64_C( 6874356853280244370),  INT64_C( 2681720383917028356),  INT64_C(  363617062994806731), -INT64_C( 7318342761473894660),
         INT64_C( 2164784578008873609), -INT64_C( 8795130151555541168), -INT64_C( 3115818213941246733),  INT64_C( 2097447710665838201) },
      { -INT64_C( 4450058466010859147),  INT64_C(  994959937184262213), -INT64_C( 3100361591938341436), -INT64_C( 1614767496023839026),
        -INT64_C( 7737328937327470021), -INT64_C( 6564158462417554836), -INT64_C( 1333872736001888935),  INT64_C( 1018187299377882946) } },
    { { -INT64_C( 4803982436490178277), -INT64_C( 1771749786168752086), -INT64_C( 7193825152807419778),  INT64_C( 1704113265784478708),
         INT64_C( 7923416425212768202),  INT64_C( 8576185165226802729),  INT64_C(  160712378890954874),  INT64_C( 6585724530256992338) },
      {  INT64_C( 9029586771277755040), -INT64_C( 3064805812496326649), -INT64_C( 8261663046046104505), -INT64_C( 5408189745344027899),
         INT64_C( 1640407638281258483),  INT64_C( 4951300961883020935), -INT64_C( 2550739608973727279), -INT64_C( 6368259520359861034) },
      { -INT64_C( 4603986968108719173),  INT64_C( 3611576490511247405),  INT64_C( 1257290549943567417), -INT64_C( 6677578309627889423),
         INT64_C( 8878362109278874169),  INT64_C( 3725100340635705518), -INT64_C( 2404072443576090197), -INT64_C(  217769111299019644) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i a = simde_mm512_loadu_epi64(test_vec[i].a);
    simde__m512i b = simde_mm512_loadu_epi64(test_vec[i].b);
    simde__m512i r = simde_mm512_xor_epi64(a, b);
    simde_test_x86_assert_equal_i64x8(r, simde_mm512_loadu_epi64(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm512_mask_xor_epi64 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int64_t src[8];
    const simde__mmask8 k;
    const int64_t a[8];
    const int64_t b[8];
    const int64_t r[8];
  } test_vec[] = {
    { {  INT64_C(   45749859996612839),  INT64_C( 2814572066373608668), -INT64_C( 6582897561231417342), -INT64_C( 8687305550852076307),
         INT64_C( 8500406415190396854),  INT64_C( 5093384555923498535),  INT64_C( 8909703965306565750),  INT64_C(  546313591099609965) },
      UINT8_C(244),
      { -INT64_C( 2585626980132258187),  INT64_C(  726882011601479497), -INT64_C( 8038622157332801693), -INT64_C( 8691410639997796728),
         INT64_C( 4087190315360841190), -INT64_C( 6654652016916703164), -INT64_C( 2859196078284653307),  INT64_C(  229866223008298134) },
      { -INT64_C( 7945590762679324780), -INT64_C( 8839772767949909393), -INT64_C( 6397194970787743473), -INT64_C( 5121297404723757619),
        -INT64_C( 3311000718894445522),  INT64_C( 1769302264377325704),  INT64_C( 4288746735372129484), -INT64_C( 9076703424935504851) },
      {  INT64_C(   45749859996612839),  INT64_C( 2814572066373608668),  INT64_C( 3983867818266324588), -INT64_C( 8687305550852076307),
        -INT64_C( 1534483321180642872), -INT64_C( 4960646585427500852), -INT64_C( 2029249597307622967), -INT64_C( 9135072113565048645) } },
    { { -INT64_C( 4389873590959520546),  INT64_C( 3847016815465829697), -INT64_C( 5127463024880002259), -INT64_C( 1350777454883832725),
        -INT64_C( 8622911936856637864), -INT64_C( 6379571547940445736), -INT64_C( 8262950684676879856), -INT64_C( 3957664130431951338) },
      UINT8_C(203),
      { -INT64_C( 3288302065678132158), -INT64_C( 6867158103737196692),  INT64_C(   29078381664489015), -INT64_C( 2093545317731299942),
         INT64_C(  800449542886982807), -INT64_C( 7919193623266236573), -INT64_C( 4301136478309459027), -INT64_C( 7828699826288344616) },
      { -INT64_C( 1617647112863104542),  INT64_C( 5065560676272060766),  INT64_C( 5928115323832446807), -INT64_C( 1137963685561880410),
         INT64_C( 7301739104183587045), -INT64_C( 8568825949546695114),  INT64_C( 3848130783034911386),  INT64_C( 4286595986196214142) },
      {  INT64_C( 4310345289044696480), -INT64_C( 1801852656978844110), -INT64_C( 5127463024880002259),  INT64_C( 1353085047476567356),
        -INT64_C( 8622911936856637864), -INT64_C( 6379571547940445736), -INT64_C( 1069598699435757257), -INT64_C( 6329859119562035034) } },
    { {  INT64_C( 2488796161233055261), -INT64_C( 6022150448521041999),  INT64_C( 4933174169979353009),  INT64_C( 8640803886955777369),
         INT64_C( 4344324970652602684), -INT64_C(  898031332366467893), -INT64_C( 3887263497667196412),  INT64_C( 7478337318462021367) },
      UINT8_C(245),
      {  INT64_C( 1608838238175268707),  INT64_C(  319276987121676509),  INT64_C( 2531375131368175488), -INT64_C( 6984319707087211591),
         INT64_C( 3257643290499388229),  INT64_C( 6514463296540085930), -INT64_C( 2908017913228894916), -INT64_C( 1228629849485806824) },
      {  INT64_C( 4402061463343995290), -INT64_C( 1223561056477925679),  INT64_C( 5177755488274713062),  INT64_C( 8115643581275557640),
        -INT64_C( 4361610958211469656),  INT64_C(  631194590693431583),  INT64_C( 5869570511533257725),  INT64_C( 5115705028593717175) },
      {  INT64_C( 3117896433371354873), -INT64_C( 6022150448521041999),  INT64_C( 7276222062497414758),  INT64_C( 8640803886955777369),
        -INT64_C( 1275353631281103379),  INT64_C( 5956698770186540981), -INT64_C( 8732406741720305983), -INT64_C( 6337227441341793105) } },
    { { -INT64_C( 1655642640894306224), -INT64_C( 1928527008934104302), -INT64_C( 1780464257288025679),  INT64_C( 4645871961807080853),
         INT64_C(  396184772487084747),  INT64_C( 7694247929610583034), -INT64_C(  876321354710525762), -INT64_C( 8490798505132667909) },
      UINT8_C(153),
      { -INT64_C( 2797138209894817876), -INT64_C( 5764892471572950838),  INT64_C( 1083398023113731424), -INT64_C( 3864555640558062810),
         INT64_C( 5039295415212501623), -INT64_C( 2395207474291048132), -INT64_C(  629670723305287568), -INT64_C( 2391099797848452888) },
      {  INT64_C( 4019150386831347461),  INT64_C( 4521628108579294052), -INT64_C( 8456435593802240804), -INT64_C( 9148392511393210172),
         INT64_C( 5071836999787413040), -INT64_C( 7700616903254299628),  INT64_C(  919268992504468373), -INT64_C( 2070738063835246265) },
      { -INT64_C( 1231608716974305111), -INT64_C( 1928527008934104302), -INT64_C( 1780464257288025679),  INT64_C( 5428023262223111138),
         INT64_C(  256112104962790471),  INT64_C( 7694247929610583034), -INT64_C(  876321354710525762),  INT64_C( 4436707353278300591) } },
    { {  INT64_C( 2479355900382364993),  INT64_C( 1111304417577308446), -INT64_C( 2590687349108723784), -INT64_C( 4039761275362786215),
        -INT64_C( 8189705280462062040),  INT64_C( 4210251584504159459),  INT64_C( 1509346594468311936), -INT64_C( 7144943817645321846) },
      UINT8_C( 14),
      { -INT64_C( 8721207303514172734),  INT64_C( 8557758222839600753),  INT64_C( 6723388035183008125), -INT64_C(  395352253016219803),
         INT64_C( 9146538221902954815), -INT64_C( 5145499759254898594),  INT64_C( 8396340341971405148),  INT64_C(  170411947343828522) },
      { -INT64_C( 3244744075203875731), -INT64_C( 7474133439378900910), -INT64_C( 5127834652165225900), -INT64_C( 7466837154034172005),
         INT64_C( 8659025868352572388), -INT64_C(  173585767618699581),  INT64_C( 4130986808766912526), -INT64_C( 3838482065090187530) },
      {  INT64_C( 2479355900382364993), -INT64_C( 1259379519748575709), -INT64_C( 1902770539115504855),  INT64_C( 7125567705071739134),
        -INT64_C( 8189705280462062040),  INT64_C( 4210251584504159459),  INT64_C( 1509346594468311936), -INT64_C( 7144943817645321846) } },
    { { -INT64_C( 7322204454426325821), -INT64_C( 4462169255779407764), -INT64_C( 6962650059328556152), -INT64_C( 5561527802563267039),
        -INT64_C( 5928710453305019577),  INT64_C(  782735190010253669),  INT64_C( 1049778440287304264),  INT64_C( 1524159294001730182) },
      UINT8_C(217),
      {  INT64_C( 4406147630084273090),  INT64_C( 6172405503935594122),  INT64_C( 9083848596339727098),  INT64_C( 7864517138114643045),
        -INT64_C( 8017162673287577708),  INT64_C( 4755687165104520592), -INT64_C(  658691503089303246),  INT64_C( 1979444766424770400) },
      { -INT64_C( 8774039058048637787), -INT64_C(  694063673707057962),  INT64_C( 7347955965608798833),  INT64_C( 5114374633758407510),
        -INT64_C( 4392929463379061619), -INT64_C( 2660713234140481191), -INT64_C( 1381242144590764183), -INT64_C( 4973730811908373486) },
      { -INT64_C( 4964763279348885657), -INT64_C( 4462169255779407764), -INT64_C( 6962650059328556152),  INT64_C( 3160832092535504691),
         INT64_C( 6031583149779954457),  INT64_C(  782735190010253669),  INT64_C( 1877732794698114651), -INT64_C( 6808980108460137614) } },
    { {  INT64_C( 8216894590175538007), -INT64_C( 8015102044995065729), -INT64_C( 8319041892314537462),  INT64_C( 8069950358681507095),
         INT64_C(  121240403729136420), -INT64_C( 1487121492127105372),  INT64_C( 4995924851354721104), -INT64_C(  940257476621986452) },
      UINT8_C(196),
      {  INT64_C( 1947321805534736039), -INT64_C( 8561651653089111504), -INT64_C( 2092489041890510734),  INT64_C( 1500721690142866054),
        -INT64_C( 8085793806064609071),  INT64_C( 2163588935650821455),  INT64_C( 4667027440343170452), -INT64_C( 1686560576050858730) },
      { -INT64_C( 2582427849531715781), -INT64_C( 3968128441716664540),  INT64_C( 5195024434345973022), -INT64_C( 4187110105943349020),
        -INT64_C(   81476173827179916), -INT64_C( 5323890722526513120),  INT64_C( 1931383667822610259),  INT64_C( 8313639412841224719) },
      {  INT64_C( 8216894590175538007), -INT64_C( 8015102044995065729), -INT64_C( 6130081342750364308),  INT64_C( 8069950358681507095),
         INT64_C(  121240403729136420), -INT64_C( 1487121492127105372),  INT64_C( 6487782212416142023), -INT64_C( 7221564986028855527) } },
    { {  INT64_C( 7515754918875348596),  INT64_C( 3232135209222413451),  INT64_C( 2229984471521781921),  INT64_C( 8541946416010364953),
        -INT64_C( 2412501877195996971), -INT64_C( 4885037687150329554),  INT64_C( 9039434068048209220),  INT64_C( 1659050078198114957) },
      UINT8_C( 33),
      { -INT64_C(  429157281734249269),  INT64_C( 3203978284322067029),  INT64_C( 7455724837908040293), -INT64_C( 2817354451018840157),
         INT64_C(  335448553716800499), -INT64_C(  917138322303370029), -INT64_C( 8711837563429726603),  INT64_C( 8377595263254839645) },
      { -INT64_C( 8340727286980282227), -INT64_C( 7636439070152493692), -INT64_C( 7697670767163939903),  INT64_C( 3969997471685310498),
        -INT64_C( 8826859548180268741), -INT64_C( 3251989183356983769), -INT64_C( 6246875829788793216), -INT64_C( 3603330975078199780) },
      {  INT64_C( 8517603223431072838),  INT64_C( 3232135209222413451),  INT64_C( 2229984471521781921),  INT64_C( 8541946416010364953),
        -INT64_C( 2412501877195996971),  INT64_C( 2421589282539805428),  INT64_C( 9039434068048209220),  INT64_C( 1659050078198114957) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i src = simde_mm512_loadu_epi64(test_vec[i].src);
    simde__m512i a = simde_mm512_loadu_epi64(test_vec[i].a);
    simde__m512i b = simde_mm512_loadu_epi64(test_vec[i].b);
    simde__m512i r = simde_mm512_mask_xor_epi64(src, test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i64x8(r, simde_mm512_loadu_epi64(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm512_maskz_xor_epi64 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde__mmask8 k;
    const int64_t a[8];
    const int64_t b[8];
    const int64_t r[8];
  } test_vec[] = {
    { UINT8_C(152),
      { -INT64_C( 8982237529053997842),  INT64_C( 3103674849943855424),  INT64_C( 1982415417451867211), -INT64_C( 9083321110428971905),
         INT64_C( 7887070877457153635),  INT64_C( 7936046385343671330),  INT64_C( 1142105130860059500), -INT64_C( 2327418192463860577) },
      {  INT64_C( 9182915906006801623), -INT64_C( 3782010632694931727), -INT64_C( 4191618534024375243),  INT64_C( 7829976987492367438),
        -INT64_C( 5664436787247809914), -INT64_C( 6275684222459520321),  INT64_C( 2362774424294093278), -INT64_C(  972614327583719042) },
      {  INT64_C(                   0),  INT64_C(                   0),  INT64_C(                   0), -INT64_C( 1344283778605633487),
        -INT64_C( 2587491794353060635),  INT64_C(                   0),  INT64_C(                   0),  INT64_C( 3257172424768237025) } },
    { UINT8_C( 14),
      {  INT64_C( 1409767669188450032),  INT64_C(  535417729420012870), -INT64_C(  117662871603022948), -INT64_C( 2123625407593156124),
         INT64_C( 2053434539035022643), -INT64_C( 3346989035654740496),  INT64_C( 5028564073104500622), -INT64_C( 4300579678214802307) },
      {  INT64_C( 4002766890185262234), -INT64_C( 2181348245731657548), -INT64_C(  911567200057529934), -INT64_C(  648602612603414614),
        -INT64_C( 7910868655864926281), -INT64_C(  981726171945503701),  INT64_C( 3172889850273926303), -INT64_C( 2970011671702574219) },
      {  INT64_C(                   0), -INT64_C( 1813713950618380814),  INT64_C(  938028879597345326),  INT64_C( 1475191305885214286),
         INT64_C(                   0),  INT64_C(                   0),  INT64_C(                   0),  INT64_C(                   0) } },
    { UINT8_C(149),
      { -INT64_C(  122716449265167314),  INT64_C( 3987427480104258177), -INT64_C( 2447171538408346827), -INT64_C( 3365926546413069491),
         INT64_C( 4949525000721972647),  INT64_C( 6082763436880293385),  INT64_C( 6273216357046166716), -INT64_C( 8772636185306919988) },
      {  INT64_C( 8870376520984048402),  INT64_C( 2542591394626541015), -INT64_C( 1180399248176107203),  INT64_C( 1283984387118967405),
        -INT64_C( 3908843060537718317), -INT64_C( 4206915619632812268), -INT64_C( 4662164243526913735),  INT64_C( 2190117534882835284) },
      { -INT64_C( 8838896490382629060),  INT64_C(                   0),  INT64_C( 3573473610564262408),  INT64_C(                   0),
        -INT64_C( 8254885696627231628),  INT64_C(                   0),  INT64_C(                   0), -INT64_C( 7483423931756404584) } },
    { UINT8_C(230),
      { -INT64_C( 7827641925076753334),  INT64_C( 7539554054630529375), -INT64_C(  798410052431249940), -INT64_C( 1276666438538543227),
         INT64_C( 2340348715907686594),  INT64_C( 7084207804521158000),  INT64_C(  821033393377053478), -INT64_C( 4272309647950633948) },
      { -INT64_C(  813422530811728427), -INT64_C( 8173422651522728262), -INT64_C( 7654040801051243466), -INT64_C( 4698359519177329658),
         INT64_C( 6158402678492002303), -INT64_C( 5386841171882060983), -INT64_C( 6509328055499932508),  INT64_C( 1734864526403044972) },
      {  INT64_C(                   0), -INT64_C( 1858918950121707547),  INT64_C( 7001998150177485274),  INT64_C(                   0),
         INT64_C(                   0), -INT64_C( 2923345196226853319), -INT64_C( 5850495760465381502), -INT64_C( 2547120826193042872) } },
    { UINT8_C(241),
      {  INT64_C( 7402477510674193255), -INT64_C( 1863195392533566616), -INT64_C( 1608300861362718454),  INT64_C( 9026368236341560280),
         INT64_C( 2030747075921039797), -INT64_C( 5213830620423384300),  INT64_C( 3226169786223897081), -INT64_C( 7548099076051449941) },
      { -INT64_C( 1412565848515727723), -INT64_C( 5042819090609471071),  INT64_C( 5906290208258063991), -INT64_C( 3559159576159619549),
         INT64_C( 3046259127309826945), -INT64_C( 5262319553509592963),  INT64_C(  968688341824409210),  INT64_C( 6482559857543185902) },
      { -INT64_C( 8439907556581526030),  INT64_C(                   0),  INT64_C(                   0),  INT64_C(                   0),
         INT64_C( 3920617936044698164),  INT64_C(   98160633604465513),  INT64_C( 2428795596282832771), -INT64_C( 3546182036552269243) } },
    { UINT8_C(168),
      { -INT64_C( 1654971735735160050), -INT64_C( 3618396837604997422), -INT64_C( 2321671188407392160), -INT64_C(  852569122218184746),
        -INT64_C( 8527980720679920445),  INT64_C( 1070063051582525133),  INT64_C( 3153274373031825801),  INT64_C( 5582450922785503646) },
      {  INT64_C( 5365668961424373301),  INT64_C( 8590950464161730174),  INT64_C( 7872605299379151040), -INT64_C(   41541984007884651),
        -INT64_C( 6690553381164190369),  INT64_C( 7788140668098870447),  INT64_C( 2478321804687601704),  INT64_C( 2196821751906784692) },
      {  INT64_C(                   0),  INT64_C(                   0),  INT64_C(                   0),  INT64_C(  812751503155928899),
         INT64_C(                   0),  INT64_C( 7119249835665306210),  INT64_C(                   0),  INT64_C( 5982033988820085802) } },
    { UINT8_C( 34),
      { -INT64_C( 7099385906741118360), -INT64_C( 7139975430881601607), -INT64_C( 7228194637974216392),  INT64_C( 1322782488553664140),
         INT64_C( 5057636448096527218),  INT64_C( 7509173144721521260),  INT64_C( 3869178845850853074), -INT64_C(  792411965939119037) },
      { -INT64_C( 7115483786937574292),  INT64_C(  849890848911280179), -INT64_C( 8527612870863474487),  INT64_C( 5429785197312153057),
        -INT64_C( 3592881174692371884), -INT64_C( 8281559221398316595), -INT64_C( 1513871340195597267),  INT64_C( 4748387116286637243) },
      {  INT64_C(                   0), -INT64_C( 7556234139783962742),  INT64_C(                   0),  INT64_C(                   0),
         INT64_C(                   0), -INT64_C( 1935407695096568927),  INT64_C(                   0),  INT64_C(                   0) } },
    { UINT8_C(139),
      {  INT64_C( 4935371154977638186), -INT64_C( 3940202631690419418),  INT64_C( 3960841935430796437), -INT64_C( 6899968217622343414),
        -INT64_C( 9198094981434153942), -INT64_C( 8237377322867269296),  INT64_C( 5165840138584555705),  INT64_C( 7019203302184899410) },
      { -INT64_C( 2992976723773572708), -INT64_C( 4056219280331761360), -INT64_C( 8039221936049258773),  INT64_C( 4421622930838481093),
        -INT64_C( 3253945696776891781), -INT64_C( 1614628432426396734), -INT64_C( 7999378326465488595), -INT64_C( 7870629585987349017) },
      { -INT64_C( 7923174882398929226),  INT64_C( 1073260354067536406),  INT64_C(                   0), -INT64_C( 7105939320326063665),
         INT64_C(                   0),  INT64_C(                   0),  INT64_C(                   0), -INT64_C(  888106185265150283) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i a = simde_mm512_loadu_epi64(test_vec[i].a);
    simde__m512i b = simde_mm512_loadu_epi64(test_vec[i].b);
    simde__m512i r = simde_mm512_maskz_xor_epi64(test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i64x8(r, simde_mm512_loadu_epi64(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm512_xor_si512(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
    { simde_x_mm512_set_epu64(UINT64_C(0xedb78aa51009d043), UINT64_C(0xf8d6e1466c80412e),
                              UINT64_C(0x8d2f88ccf8d072d2), UINT64_C(0xd42ce380801d56eb),
                              UINT64_C(0x4f7a9f9f877cf207), UINT64_C(0x9ebf29784a068fec),
                              UINT64_C(0x14e312298b327bb2), UINT64_C(0xa4cae715b44968c3)),
      simde_x_mm512_set_epu64(UINT64_C(0xf857a2af697b20a2), UINT64_C(0xaff5e6cb29617e18),
                              UINT64_C(0x91849348f298760e), UINT64_C(0x1c2d26b7c60b0e1b),
                              UINT64_C(0x27f2529cc5e0d6da), UINT64_C(0x947c2755a9d7153c),
                              UINT64_C(0xab6596dc73591502), UINT64_C(0x6cb918d0cf4b1739)),
      simde_x_mm512_set_epu64(UINT64_C(0x15e0280a7972f0e1), UINT64_C(0x5723078d45e13f36),
                              UINT64_C(0x1cab1b840a4804dc), UINT64_C(0xc801c537461658f0),
                              UINT64_C(0x6888cd03429c24dd), UINT64_C(0x0ac30e2de3d19ad0),
                              UINT64_C(0xbf8684f5f86b6eb0), UINT64_C(0xc873ffc57b027ffa)) },
    { simde_x_mm512_set_epu64(UINT64_C(0x7fb6b2cc6cfe2095), UINT64_C(0x2b06563737a7554c),
                              UINT64_C(0x20e62cd04a81579d), UINT64_C(0x0c22d8b7c6b9afd0),
                              UINT64_C(0x7778227f653f51e3), UINT64_C(0x0c5d93feab9aa991),
                              UINT64_C(0x45c2fff1a0832972), UINT64_C(0xc6fefc42436c6b46)),
      simde_x_mm512_set_epu64(UINT64_C(0xe44de61268819c50), UINT64_C(0xec0f232714f1db42),
                              UINT64_C(0x1de6bf50b7448d81), UINT64_C(0x326b4ae523cd1df1),
                              UINT64_C(0x89856b1e9b31c8bb), UINT64_C(0xb4ab21a1f0881bd7),
                              UINT64_C(0x3d731298d95d6f21), UINT64_C(0xffc08a64375bf884)),
      simde_x_mm512_set_epu64(UINT64_C(0x9bfb54de047fbcc5), UINT64_C(0xc709751023568e0e),
                              UINT64_C(0x3d009380fdc5da1c), UINT64_C(0x3e499252e574b221),
                              UINT64_C(0xfefd4961fe0e9958), UINT64_C(0xb8f6b25f5b12b246),
                              UINT64_C(0x78b1ed6979de4653), UINT64_C(0x393e7626743793c2)) },
    { simde_x_mm512_set_epu64(UINT64_C(0xdcc26142d37fb5e3), UINT64_C(0x63c9856267e9999a),
                              UINT64_C(0xbf8d48fd4246771e), UINT64_C(0xe34d12aa2d211228),
                              UINT64_C(0x38f5235239303226), UINT64_C(0x264f8a7d4f0c7a44),
                              UINT64_C(0x3e3aa9db569a6f79), UINT64_C(0x47f95a5dbaf7c3fe)),
      simde_x_mm512_set_epu64(UINT64_C(0x24f03a01ff0412a4), UINT64_C(0x5e69b3ae6e914583),
                              UINT64_C(0x61a1b3291cf72225), UINT64_C(0x5b7b6dafc3dcc6f8),
                              UINT64_C(0x643061d1edb662f2), UINT64_C(0x3eebdc3f8f4056db),
                              UINT64_C(0x11c4727a73fc286e), UINT64_C(0x561b4fb689bf8f8c)),
      simde_x_mm512_set_epu64(UINT64_C(0xf8325b432c7ba747), UINT64_C(0x3da036cc0978dc19),
                              UINT64_C(0xde2cfbd45eb1553b), UINT64_C(0xb8367f05eefdd4d0),
                              UINT64_C(0x5cc54283d48650d4), UINT64_C(0x18a45642c04c2c9f),
                              UINT64_C(0x2ffedba125664717), UINT64_C(0x11e215eb33484c72)) },
    { simde_x_mm512_set_epu64(UINT64_C(0xa3db7414654163c1), UINT64_C(0x20295e6408f5e85e),
                              UINT64_C(0x6544618e6bd8d65e), UINT64_C(0x5e62e456253a5970),
                              UINT64_C(0x35200d7cedf89e63), UINT64_C(0x3a187ccb9bdbc4ff),
                              UINT64_C(0x8c83780e03d2ec0a), UINT64_C(0x25da10ac4ca3d5f3)),
      simde_x_mm512_set_epu64(UINT64_C(0xb4da361a4ff83c1b), UINT64_C(0x7f54a2cd02321d93),
                              UINT64_C(0x2222e01cb6f3b71d), UINT64_C(0x48d83f4fe210f439),
                              UINT64_C(0x30b5939d74f88fb5), UINT64_C(0x08451aa5c5aafb71),
                              UINT64_C(0x0908270bde506014), UINT64_C(0x14d2968168fbf701)),
      simde_x_mm512_set_epu64(UINT64_C(0x1701420e2ab95fda), UINT64_C(0x5f7dfca90ac7f5cd),
                              UINT64_C(0x47668192dd2b6143), UINT64_C(0x16badb19c72aad49),
                              UINT64_C(0x05959ee1990011d6), UINT64_C(0x325d666e5e713f8e),
                              UINT64_C(0x858b5f05dd828c1e), UINT64_C(0x3108862d245822f2)) },
    { simde_x_mm512_set_epu64(UINT64_C(0xb9888f8a15c6f599), UINT64_C(0xdae6980a3c15b8d5),
                              UINT64_C(0x17114f3e96d162e1), UINT64_C(0xaa441d9be0eb3305),
                              UINT64_C(0x7328bea0eddeb5b8), UINT64_C(0x38d955208ba6ab2c),
                              UINT64_C(0xd5a6f9d82f72b047), UINT64_C(0x468d076219769ecc)),
      simde_x_mm512_set_epu64(UINT64_C(0xf5fce010c130811a), UINT64_C(0x5b4c8bc96595cc6f),
                              UINT64_C(0x9ec90bdb77fd0d0f), UINT64_C(0xe13db6113bafebb0),
                              UINT64_C(0xbe6dfb35371e254d), UINT64_C(0xee5939c207b9c26b),
                              UINT64_C(0x7c3ef03f0a2d4864), UINT64_C(0xe807e98806d6b3fa)),
      simde_x_mm512_set_epu64(UINT64_C(0x4c746f9ad4f67483), UINT64_C(0x81aa13c3598074ba),
                              UINT64_C(0x89d844e5e12c6fee), UINT64_C(0x4b79ab8adb44d8b5),
                              UINT64_C(0xcd454595dac090f5), UINT64_C(0xd6806ce28c1f6947),
                              UINT64_C(0xa99809e7255ff823), UINT64_C(0xae8aeeea1fa02d36)) },
    { simde_x_mm512_set_epu64(UINT64_C(0x8234186be169c857), UINT64_C(0x6e3be8c42ba36d9a),
                              UINT64_C(0x9eebbbe6bd8adb2a), UINT64_C(0x6ce901141909d2cf),
                              UINT64_C(0x35459cc296fca858), UINT64_C(0x1a7d575fa8651237),
                              UINT64_C(0x4b008fe37abafacd), UINT64_C(0xf35eba645c1d884d)),
      simde_x_mm512_set_epu64(UINT64_C(0x7ea8964c6c682a7c), UINT64_C(0x8b6605b470502155),
                              UINT64_C(0x4b16327f96bf6e87), UINT64_C(0xae618aa0114ea6c6),
                              UINT64_C(0x3c1572ee53b136fa), UINT64_C(0xacef14edc9d741a1),
                              UINT64_C(0x96f4d64c8555893a), UINT64_C(0x1fb0ce0c9ed59cf4)),
      simde_x_mm512_set_epu64(UINT64_C(0xfc9c8e278d01e22b), UINT64_C(0xe55ded705bf34ccf),
                              UINT64_C(0xd5fd89992b35b5ad), UINT64_C(0xc2888bb408477409),
                              UINT64_C(0x0950ee2cc54d9ea2), UINT64_C(0xb69243b261b25396),
                              UINT64_C(0xddf459afffef73f7), UINT64_C(0xecee7468c2c814b9)) },
    { simde_x_mm512_set_epu64(UINT64_C(0xd4967d973e742c64), UINT64_C(0xcb3e880be1980939),
                              UINT64_C(0xc418352686ff3548), UINT64_C(0xdb9cc81b4939caef),
                              UINT64_C(0x99908ab055e14bf0), UINT64_C(0xd01deeb18277fd8f),
                              UINT64_C(0xe1f43dbe1a24fb3a), UINT64_C(0xdaa3b7846091d1be)),
      simde_x_mm512_set_epu64(UINT64_C(0x475003e212ada19b), UINT64_C(0x490bdb33ee5d5470),
                              UINT64_C(0x61249881556eac3a), UINT64_C(0xbf42ccd4a27e5259),
                              UINT64_C(0x613173560a9ec8e6), UINT64_C(0x8736f836c78d1256),
                              UINT64_C(0xa4248bd9dac1f2cc), UINT64_C(0x7008605d8072d787)),
      simde_x_mm512_set_epu64(UINT64_C(0x93c67e752cd98dff), UINT64_C(0x823553380fc55d49),
                              UINT64_C(0xa53cada7d3919972), UINT64_C(0x64de04cfeb4798b6),
                              UINT64_C(0xf8a1f9e65f7f8316), UINT64_C(0x572b168745faefd9),
                              UINT64_C(0x45d0b667c0e509f6), UINT64_C(0xaaabd7d9e0e30639)) },
    { simde_x_mm512_set_epu64(UINT64_C(0xc6500379d74d1915), UINT64_C(0x2deb735fa56e277e),
                              UINT64_C(0xc2e0f463b67c41f4), UINT64_C(0x8f539a5e01d0c88f),
                              UINT64_C(0x68e4935ea747c9c2), UINT64_C(0xdc21f9b373f8b465),
                              UINT64_C(0xf3592239b25cb40f), UINT64_C(0xf4139e2d72ff74c8)),
      simde_x_mm512_set_epu64(UINT64_C(0x7e1193710ce44a9c), UINT64_C(0x253a368d6b9cc286),
                              UINT64_C(0x310c01bdff0560df), UINT64_C(0xe73fc91eec559d39),
                              UINT64_C(0xc18711aa058fbe1a), UINT64_C(0x80fe26999b91720a),
                              UINT64_C(0x12959cadf8f60c1b), UINT64_C(0xae1ad9214abbd4ef)),
      simde_x_mm512_set_epu64(UINT64_C(0xb8419008dba95389), UINT64_C(0x08d145d2cef2e5f8),
                              UINT64_C(0xf3ecf5de4979212b), UINT64_C(0x686c5340ed8555b6),
                              UINT64_C(0xa96382f4a2c877d8), UINT64_C(0x5cdfdf2ae869c66f),
                              UINT64_C(0xe1ccbe944aaab814), UINT64_C(0x5a09470c3844a027)) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_xor_si512(test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_u64(r, ==, test_vec[i].r);
  }

  return 0;
}

SIMDE_TEST_FUNC_LIST_BEGIN
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_xor_ps)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mask_xor_ps)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_maskz_xor_ps)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_xor_pd)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mask_xor_pd)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_maskz_xor_pd)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_xor_epi32)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mask_xor_epi32)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_maskz_xor_epi32)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_xor_epi64)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_mask_xor_epi64)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_maskz_xor_epi64)
  SIMDE_TEST_FUNC_LIST_ENTRY(mm512_xor_si512)
SIMDE_TEST_FUNC_LIST_END

#include <test/x86/avx512/test-avx512-footer.h>
